(*
 * V-variable Fractal Generator
 *
 * File:   VVariableFractalMain.pas
 * Author: Bram Kuijvenhoven (bram (at) kuijvenhoven (dot) net)
 * Descr.: Main part of program code: Main window & program logic
 *
 *)

unit VVariableFractalMain;

{$mode objfpc}{$H+}

interface

uses
  // FPC
  Classes, SysUtils, TypInfo, Contnrs,
  // LCL
  LResources, Forms, Controls, Graphics, Dialogs, Buttons, StdCtrls, ComCtrls, LCLProc,
    RTTICtrls, Grids, RTTIGrids, Menus, ActnList,
  // Lazarus
  PropEdits,
  // GLScene
  gllclviewer, glscene, globjects, glcadencer, gltexture, glmisc, GLGraphics, GLColor,
    GLSimpleNavigation, GLGeomObjects, GLContext, GLProxyObjects,
  VectorGeometry, VectorTypes,
  // Custom
  ObjectEditor, AboutFrm;

type

  { Forward declarations }
  TTransformation = class;
  TBuffer = class;
  TIFS = class;
  TRootComponent = class;
  TVVariableFractal = class;
  
  (*
   * GUI
   *)
   
  TLambdaInt = 0..20;
  TStickyMode = (smNone, smIdentity, smMove);
  TAnimationMode = (amLast, amFirst, amAll); // which transformation gets lambda param?

  { TMainForm }
  TMainForm = class(TForm)
    AboutAction: TAction;
    AnimateCheckBox: TTICheckBox;
    AniModeTIComboBox: TTIComboBox;
    EditAction: TAction;
    FractalSepMenuItem: TMenuItem;
    EditMenuItem: TMenuItem;
    BufDistLabel: TLabel;
    AnimationGroupBox: TGroupBox;
    HelpMenuItem: TMenuItem;
    AboutMenuItem: TMenuItem;
    StickyDistTIFloatSpinEdit: TTIFloatSpinEdit;
    StickyModeLabel: TLabel;
    StickyModeTIComboBox: TTIComboBox;
    StikcyGroupBox: TGroupBox;
    IterationGroupBox: TGroupBox;
    LambdaTrackBar: TTITrackBar;
    StepAction: TAction;
    InitAction: TAction;
    FractalMenuItem: TMenuItem;
    InitMenuItem: TMenuItem;
    StepMenuItem: TMenuItem;
    SaveMenuItem: TMenuItem;
    SaveAction: TAction;
    ExitAction: TAction;
    ActionList: TActionList;
    GLSimpleNavigation: TGLSimpleNavigation;
    MainMenu: TMainMenu;
    FileMenuItem: TMenuItem;
    ExitMenuItem: TMenuItem;
    BufDistTIFloatSpinEdit: TTIFloatSpinEdit;
    VVFListView: TListView;
    VVFListLabel: TLabel;
    StepButton: TButton;
    InitButton: TButton;
    GLCadencer: TGLCadencer;
    GLCamera: TGLCamera;
    GLScene: TGLScene;
    GLSceneViewer: TGLSceneViewer;
    // Form events
    procedure AboutMenuItemClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: boolean);
    procedure FormDestroy(Sender: TObject);
    // Action events
    procedure SaveActionExecute(Sender: TObject);
    procedure ExitActionExecute(Sender: TObject);
    procedure InitActionExecute(Sender: TObject);
    procedure StepActionExecute(Sender: TObject);
    procedure EditActionExecute(Sender: TObject);
    // VVFListView events
    procedure VVFListViewColumnClick(Sender: TObject; Column: TListColumn);
    procedure VVFListViewDblClick(Sender: TObject);
    procedure VVFListViewSelectItem(Sender: TObject; Item: TListItem;  Selected: Boolean);
    // Misc events
    procedure GLCadencerProgress(Sender: TObject; const deltaTime, newTime: Double);
    procedure NavigationCustomAction(Sender: TGLSimpleNavigationKeyCombination; Shift: TShiftState; X, Y: Integer);
    // PropHook events
    procedure PropHookModified(Sender: TObject);
    procedure PropHookRevert(Instance:TPersistent; PropInfo:PPropInfo);
    procedure PropHookRefreshPropertyValues;
    procedure PropHookPersistentAdded(APersistent: TPersistent; Select: boolean);
    procedure PropHookPersistentDeleting(APersistent: TPersistent);
    procedure PropHookComponentRenamed(AComponent: TComponent);
  private
    FOldNavX, FOldNavY: integer;
    FAnimate: boolean;
    FAnimationMode: TAnimationMode;
    FBufferDistance: single;
    FLambda: single;
    FOpenFileName: string;
    FStickyDistance: single;
    FStickyMode: TStickyMode;
    FVVariableFractal:TVVariableFractal;
    FDataModule:TDataModule;
    FObjectEditor:TObjectEditorForm;
    FAboutForm:TAboutForm;
    FPropertyEditorHook:TPropertyEditorHook;
    FUpdatingVVF:boolean;
    function GetLambdaInt: TLambdaInt;
    procedure SetBufferDistance(const AValue: single);
    procedure SetLambda(const AValue: single);
    procedure SetLambdaInt(const AValue: TLambdaInt);
    procedure SetOpenFileName(const AValue: string);
    procedure SetStickyDistance(const AValue: single);
    procedure SetStickyMode(const AValue: TStickyMode);
  protected
    procedure ApplyLambda;

    procedure FindComponentClass(Reader: TReader; const AClassName: string; var ComponentClass: TComponentClass);
    procedure SaveDataModule(const FileName:string; DataModule:TDataModule);
    function LoadDataModule(const FileName:string):TDataModule;

    function GetSelectedVVF:TVVariableFractal;
    procedure UpdateVVFView(resetView:boolean);
    procedure RefreshVVFList;
  published
    { public declarations }
    property Lambda:single read FLambda write SetLambda;
    property LambdaInt:TLambdaInt read GetLambdaInt write SetLambdaInt; // useful for TTITrackBar
    property OpenFileName:string read FOpenFileName write SetOpenFileName;
    property Animate:boolean read FAnimate write FAnimate;
    property BufferDistance:single read FBufferDistance write SetBufferDistance;
    property StickyDistance:single read FStickyDistance write SetStickyDistance;
    property StickyMode:TStickyMode read FStickyMode write SetStickyMode;
    property AnimationMode:TAnimationMode read FAnimationMode write FAnimationMode;
  end;
  
  (*
   * Persistence Framework additions
   *)
   
  (*
   * TComponentReferences is a TCollection descendant holding references to TComponents --- streaming & propediting safe
   * It does request FreeNotification of referenced TComponents as to keep the references valid
   * To restrict the kind of components that can be referenced, use
   *   TComponentReferences.Create(AOwner:TPersistent; AComponentClass:TComponentClass);
   * For propediting, register TComponentReferencePropertyEditor for TComponentReferenceItem.Component:TComponent
   *)

  TComponentReferences = class;
  TComponentReferenceItem = class;

  { TFreeNotificationSink }

  TFreeNotificationSink = class(TComponent)
  private
    FCollection: TComponentReferences;
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(AOwner:TComponent; ACollection:TComponentReferences);
    property Collection:TComponentReferences read FCollection;
  end;

  { TComponentReferences }
  
  TReferenceAddedEvent = procedure(ACompRefs:TComponentReferences; AComponent: TComponent) of object;
  TReferenceRemovingEvent = procedure(ACompRefs:TComponentReferences; AComponent: TComponent) of object;

  TComponentReferences = class(TOwnedCollection)
  private
    FComponentClass: TComponentClass;
    FFreeNotificationSink: TFreeNotificationSink;
    // event handlers
    FOnReferenceAdded: TReferenceAddedEvent;
    FOnReferenceRemoving: TReferenceAddedEvent;
  protected
    function GetItem(index: integer): TComponentReferenceItem;
    procedure SetItem(index: integer; const AValue: TComponentReferenceItem);
    property FreeNotificationSink:TFreeNotificationSink read FFreeNotificationSink;
    // called by TComponentReferenceItems
    procedure NotifyReferenceAdded(AComponent:TComponent);
    procedure NotifyReferenceRemoving(AComponent:TComponent);
  public
    constructor Create(AOwner:TPersistent);
    constructor Create(AOwner:TPersistent; AComponentClass:TComponentClass);
    destructor Destroy; override;
    function Add:TComponentReferenceItem;
    property Items[index:integer]:TComponentReferenceItem read GetItem write SetItem; default;
    property ComponentClass:TComponentClass read FComponentClass; // class of components that can be referenced by contained TComponentReferenceItems
    property OnReferenceAdded:TReferenceAddedEvent read FOnReferenceAdded write FOnReferenceAdded;
    property OnReferenceRemoving:TReferenceAddedEvent read FOnReferenceRemoving write FOnReferenceRemoving;
  end;

  { TComponentReferenceItem }

  TComponentReferenceItem = class(TCollectionItem)
  private
    FComponent:TComponent;
    function GetCollection: TComponentReferences;
    function GetComponentClass: TComponentClass;
    procedure SetCollection(const AValue: TComponentReferences);
    procedure SetComponent(const AValue: TComponent);
  protected
    function GetDisplayName:string; override; // overriden from TCollectionItem
    procedure AttachComponent;
    procedure DetachComponent;
  public
    constructor Create(AOwner:TCollection); override; // overriden from TCollectionItem
    destructor Destroy; override;
    procedure Assign(source:TPersistent); override; // overriden from TCollectionItem
    property Collection:TComponentReferences read GetCollection write SetCollection;
    property ComponentClass:TComponentClass read GetComponentClass;
  published
    property Component:TComponent read FComponent write SetComponent; // the component referenced
  end;
  
  { TComponentReferencePropertyEditor }
  TComponentReferencePropertyEditor = class(TPersistentPropertyEditor)
  public
    procedure GetValues(Proc: TGetStringProc); override; // lists only those components that are of the item's ComponentClass
  end;
  
  (*
   * GLScene additions
   *)
   
  { TGLLimitProxyObject }

  TGLLimitProxyObject = class(TGLProxyObject)
  private
    FMinimalRenderSize: single;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure DoRender(var ARci : TRenderContextInfo; ARenderSelf, ARenderChildren : Boolean); override;
  published
    // proxy will only render MasterObject if some unit vector is longer than MinimalRenderSize in model space
    property MinimalRenderSize:single read FMinimalRenderSize write FMinimalRenderSize;
  end;

  (*
   * V-variable Fractal Abstraction Classes
   *
   * Note: We descend most classes from TComponent to support persistence and property editing
   *)
   
  { Array types }
  TTransformationArray = array of TTransformation;
  TBufferArray = array of TBuffer;
  TIFSArray = array of TIFS;
  TIntegerArray = array of integer;
  
  T3DShape = (
    shTetraeder,
    shCube,
    shLine
  );

  { TBuffer }
  TBuffer = class
  public
    procedure Clear(); virtual; abstract; // empty buffer
    procedure Initial(index:integer; shape:T3DShape; transform:TTransformation); virtual; abstract; // fill with initial shape; (0-based) index can be used to differentiate
    procedure AddTransformedBuf(const transform:TTransformation; const lambda:single; const buffer:TBuffer); virtual; abstract; // apply transformation and add to this buffer
  end;
  
  { TBufferFactory }
  TBufferFactory = class(TComponent)
  private
  public
    // override these in implementations
    function  GetBuffer():TBuffer; virtual; abstract;         // get a new buffer
    procedure ReleaseBuffer(buf:TBuffer); virtual; abstract;  // release a buffer to the BufferFactory
    // utility methods
    procedure GetBuffers(var bufArray:TBufferArray);
    procedure ReleaseBuffers(bufs:TBufferArray);
  published
  end;

  TTransformationChangeEvent = procedure(ATransformation:TObject) of object;

  { TTransformation }
  TTransformation = class(TComponent)
  private
    FMatrix:TMatrix;
    FNeedsCalcMatrix:boolean;
    FChangeNotifyList:TMethodList; // created on first use (i.e. call to AddChangeNotification)
    function GetMatrix: TMatrix;
  protected
    procedure CalcMatrix(lambda:single; out Mat:TMatrix); virtual; abstract; // implement in descendants; lambda = 0: return identity; lambda = 1: return actual matrix
    procedure TransformationChanged; // call this whenever the matrix needs a recalculation
  public
    destructor Destroy; override;
    procedure AddChangeNotification(AEvent:TTransformationChangeEvent);
    procedure RemoveChangeNotification(AEvent:TTransformationChangeEvent);
    function MatrixInterp(lambda:single):TMatrix;
    property Matrix:TMatrix read GetMatrix;
  end;

  { TIFS }
  TIFS = class(TComponent)
  private
    FTransforms: TComponentReferences;
    function GetTransformation(i: integer): TTransformation;
    function GetTransformationCount: integer;
    procedure SetTransformation(i: integer; const AValue: TTransformation);
    procedure SetTransforms(const AValue: TComponentReferences);
  public
    constructor Create(AOwner:TComponent); override;
    constructor Create(AOwner:TComponent; const transforms:array of TTransformation);
    destructor Destroy; override;
    property Transformation[i:integer]:TTransformation read GetTransformation write SetTransformation; // 0-based indexing
    property TransformationCount:integer read GetTransformationCount;                                  // number of transformations (M)
  published
    property Transforms:TComponentReferences read FTransforms write SetTransforms ;
  end;

  { TSelector } { or: distributions on {0,..,maxVal-1}^Count }
  TSelector = class(TComponent)
  public
    function Sample(count, maxVal: integer):TIntegerArray; virtual; abstract; // returns a sample of count values in the range 0..maxVal-1
  end;

  { TFunctionTree & TGrove }
  PFunctionTree = ^TFunctionTree;
  TFunctionTree = record
    IFSChoice:integer;            // choice of IFS
    BufferChoices:TIntegerArray;  // choice of input buffers
  end;
  
  PGrove = ^TGrove;
  TGrove = record
    Trees:array of TFunctionTree; // TFunctionTrees; one per output buffer
    InputBuffers:TBufferArray;    // references to input buffers
  end;

  { TVVariableFractal }
  TVVariableFractal = class(TComponent)
  private
    FIFSs:TComponentReferences;
    FIFSSelector:TSelector;
    FBuffers:TBufferArray;
    FBufferSelector:TSelector;
    FBufferFactory:TBufferFactory;
    FInitialized:boolean;
    FInitialShape: T3DShape;
    FInitialTransform: TTransformation;
    FGroveList: TFPList;
    FIteration: integer;
    function GetBuffer(i: integer): TBuffer;
    function GetBufferCount: integer;
    function GetGrove(AIteration: integer): PGrove;
    function GetIFS(i: integer): TIFS;
    function GetIFSCount: integer;
    function GetInputBuffer(AIteration, i: integer): TBuffer;
    function GetInputBuffers(AIteration: integer): TBufferArray;
    function GetOutputBuffer(AIteration, i: integer): TBuffer;
    function GetOutputBuffers(AIteration: integer): TBufferArray;
    procedure SetBufferCount(const AValue: integer);
    procedure SetIFSs(const AValue: TComponentReferences);
  protected
    procedure FreeBuffers; // frees all buffers in FBuffers and grove list
    function AddGrove():PGrove;
    function LastGrove():PGrove;
    procedure FreeGroves; // frees all groves, but not the grove list
  public
    constructor Create(AOwner:TComponent); override;
    constructor Create(AOwner:TComponent; bufCount:integer; const bufFactory:TBufferFactory; const ifsArray:array of TIFS; const bufSel, ifsSel:TSelector);
    destructor Destroy; override;
    
    procedure Init();
    procedure Iterate(doRender:boolean);
    procedure Render();
    procedure Render(lambda:single); // 0 <= lambda <= 1; useful for animations
    procedure Render(AIteration:integer; lambda:single); // re-do rendering of a previous iteration
  
    property Initialized:boolean read FInitialized;
    property Iteration:integer read FIteration; // 0-based iteration number
    
    property IFS[i:integer]:TIFS read GetIFS;
    property IFSCount:integer read GetIFSCount;

    property Buffer[i:integer]:TBuffer read GetBuffer; // current output buffer
    property InputBuffer[AIteration,i:integer]:TBuffer read GetInputBuffer;
    property OutputBuffer[AIteration,i:integer]:TBuffer read GetOutputBuffer;
    property InputBuffers[AIteration:integer]:TBufferArray read GetInputBuffers;
    property OutputBuffers[AIteration:integer]:TBufferArray read GetOutputBuffers;

    property Grove[AIteration:integer]:PGrove read GetGrove; // 0 <= AIteration < Iteration; there is one grove per iteration
  published
    property IFSs:TComponentReferences read FIFSs write SetIFSs;
    property IFSSelector:TSelector read FIFSSelector write FIFSSelector;
    property BufferSelector:TSelector read FBufferSelector write FBufferSelector;
    property BufferFactory:TBufferFactory read FBufferFactory write FBufferFactory;
    property BufferCount:integer read GetBufferCount write SetBufferCount;
    property InitialShape:T3DShape read FInitialShape write FInitialShape;
    property InitialTransform:TTransformation read FInitialTransform write FInitialTransform;
  end;
  
  (*
   * Implementation Classes
   *)

  { TCompoundTransformation }
  TCompoundTransformation = class(TTransformation)
  private
    FTransformations: TComponentReferences;
    procedure SetTransformations(const AValue: TComponentReferences);
  protected
    procedure CalcMatrix(lambda:single; out Mat:TMatrix); override;
    procedure ATransformationChanged(Sender:TObject);
    procedure TransformationAdded(ACompRefs: TComponentReferences; AComponent:TComponent);
    procedure TransformationRemoving(ACompRefs: TComponentReferences; AComponent:TComponent);
  public
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
  published
    property Transformations:TComponentReferences read FTransformations write SetTransformations;
  end;

  { TPointContraction }
  TPointContraction = class(TTransformation, IGLCoordinatesUpdateAble)
  private
    FFactor:single;         // contraction factor
    FPoint: TGLCoordinates; // center of contraction in 3D-space
    procedure SetFactor(const AValue: single);
    procedure SetPoint(const AValue: TGLCoordinates);
  protected
    procedure CalcMatrix(lambda:single; out Mat:TMatrix); override;
    procedure CoordinateChanged(Sender: TGLCoordinates);
  public
    constructor Create(AOwner:TComponent); override;
    constructor Create(AOwner:TComponent; aPoint:TVector; aFactor:single);
    destructor Destroy; override;
  published
    property Point:TGLCoordinates read FPoint write SetPoint;
    property Factor:single read FFactor write SetFactor;
  end;
  
  { TTranslation }
  TTranslation = class(TTransformation, IGLCoordinatesUpdateAble)
  private
    FTranslation: TGLCoordinates;
    procedure SetTranslation(const AValue: TGLCoordinates);
  protected
    procedure CalcMatrix(lambda:single; out Mat:TMatrix); override;
    procedure CoordinateChanged(Sender: TGLCoordinates);
  public
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
  published
    property Translation:TGLCoordinates read FTranslation write SetTranslation;
  end;
  
  { TRotation }
  TRotation = class(TTransformation, IGLCoordinatesUpdateAble)
  private
    FAngle: single;
    FAxis: TGLCoordinates;
    procedure SetAngle(const AValue: single);
    procedure SetAxis(const AValue: TGLCoordinates);
  protected
    procedure CalcMatrix(lambda:single; out Mat:TMatrix); override;
    procedure CoordinateChanged(Sender: TGLCoordinates);
  public
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
  published
    property Axis:TGLCoordinates read FAxis write SetAxis; // axis of rotation
    property Angle:single read FAngle write SetAngle; // angle of rotation (in degrees!)
  end;
  
  { TScaling }
  
  TScaling = class(TTransformation, IGLCoordinatesUpdateAble)
  private
    FScale: TGLCoordinates;
    procedure SetScale(const AValue: TGLCoordinates);
  protected
    procedure CalcMatrix(lambda:single; out Mat:TMatrix); override;
    procedure CoordinateChanged(Sender: TGLCoordinates);
  public
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
  published
    property Scale:TGLCoordinates read FScale write SetScale;
  end;


  T3DGLSceneBufferFactory = class;
  
  { T3DGLSceneBuffer }
  T3DGLSceneBuffer = class(TBuffer)
  private
    FBufferFactory: T3DGLSceneBufferFactory;
    FObject:TGLBaseSceneObject; // container GLScene object for this buffer
    FProxyCount:integer;
  protected
    function GetProxy():TGLLimitProxyObject; // returns a new proxy object for this buffer
  public
    constructor Create(ABufferFactory:T3DGLSceneBufferFactory);
    destructor Destroy; override;
    procedure Clear(); override;
    procedure Initial(index:integer; shape:T3DShape; transform:TTransformation); override;
    procedure AddTransformedBuf(const transform:TTransformation; const lambda:single; const buffer:TBuffer); override;
    property ProxyCount:integer read FProxyCount; // it is recommended not to free this Buffer while ProxyCount > 0
    property BufferFactory:T3DGLSceneBufferFactory read FBufferFactory;
  end;
  
  { T3DGLSceneBufferFactory }
  T3DGLSceneBufferFactory = class(TBufferFactory)
  protected
    FProxiedBufs:TFPObjectList; // list of T3DGLSceneBuffers that are proxied and will be freed on destruction or whenever FBufferCount reaches 0
    FBufferCount:integer; // (number of buffers created in GetBuffer) minus (number of buffers released in ReleaseBuffer)
  public
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
    function GetBuffer():TBuffer; override;
    procedure ReleaseBuffer(buf:TBuffer); override;
  end;

  { TIIDUniformSelector }
  TIIDUniformSelector = class(TSelector)
  public
    function Sample(count, maxVal: integer):TIntegerArray; override;
  end;
  
  { TIIDWeightedSelector }
  TIIDWeightedSelector = class(TSelector)
  private
    FWeights:TIntegerArray;
    FTotalWeight:integer;
  public
    procedure SetWeights(w:TIntegerArray);
    function Sample(count, maxVal: integer):TIntegerArray; override;
    // TODO: publish weights
  end;

  (*
   * Property Editors
   *)
   
  { TComponentListPropertyEditor }

  TComponentListPropertyEditor = class(TListPropertyEditor)
  private
    function GetComponentList:TComponentList;
  protected
    function ReadElementCount: integer; override;
    function ReadElement(Index: integer): TPersistent; override;
    function GetElementAttributes(Element: TListElementPropertyEditor): TPropertyAttributes; override;
    function GetElementName(Element: TListElementPropertyEditor):shortstring; override;
    procedure GetElementProperties(Element: TListElementPropertyEditor; Proc: TGetPropEditProc); override;
    function GetElementValue(Element: TListElementPropertyEditor): ansistring; override;
    procedure GetElementValues(Element: TListElementPropertyEditor; Proc: TGetStringProc); override;
    procedure SetElementValue(Element: TListElementPropertyEditor; NewValue: ansistring); override;
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure Edit; override;
  end;

  (* Proof-of-concept TComponent-based streaming & propediting
   *
   * Purpose is to have
   * - a TRootComponent with TChildOne's and TChildTwo's as children (this is the ownership relation)
   * - references between the various TComponent descendants
   * - a list/collection of TComponent references
   *)
   
  TChildOne = class;
  TChildTwo = class;

  { TRootComponent }

  TRootComponent = class(TComponent)
  private
    FChildCollection: TComponentReferences;
    FChildOne: TChildOne;
    FChilds: TComponentList;
    FChildTwo: TChildTwo;
    FRootInt: integer;
    FRootString: string;
    function GetChildOne: TChildOne;
    function GetChildTwo: TChildTwo;
    procedure SetChildCollection(const AValue: TComponentReferences);
    procedure SetChildOne(const AValue: TChildOne);
    procedure SetChilds(const AValue: TComponentList);
    procedure SetChildTwo(const AValue: TChildTwo);
    procedure SetRootInt(const AValue: integer);
    procedure SetRootString(const AValue: string);
  public
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
  
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
  published
    property RootInt:integer read FRootInt write SetRootInt;
    property RootString:string read FRootString write SetRootString;
    property ChildOne:TChildOne read GetChildOne write SetChildOne;
    property ChildTwo:TChildTwo read GetChildTwo write SetChildTwo;
    property Childs:TComponentList read FChilds write SetChilds;
    property ChildCollection:TComponentReferences read FChildCollection write SetChildCollection;
  end;
  
  { TChildOne }

  TChildOne = class(TComponent)
  private
    FChildOneInt: integer;
    FChildOneString: string;
    FChildTwo: TChildTwo;
    procedure SetChildOneInt(const AValue: integer);
    procedure SetChildOneString(const AValue: string);
    procedure SetChildTwo(const AValue: TChildTwo);
  public
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
  published
    property ChildOneInt:integer read FChildOneInt write SetChildOneInt;
    property ChildOneString:string read FChildOneString write SetChildOneString;
    property ChildTwo:TChildTwo read FChildTwo write SetChildTwo;
  end;
  
  { TChildTwo }

  TChildTwo = class(TComponent)
  private
    FChildTwoInt: integer;
    FChildTwoString: string;
    procedure SetChildTwoInt(const AValue: integer);
    procedure SetChildTwoString(const AValue: string);
  published
    property ChildTwoInt:integer read FChildTwoInt write SetChildTwoInt;
    property ChildTwoString:string read FChildTwoString write SetChildTwoString;
  end;
  
var
  MainForm: TMainForm;

implementation

uses
  Math, ObjInspStrConsts, GLLazarusRegister, OpenGL1x;

{ Utility }

procedure DebugClear;
begin
  //MainForm.DebugMemo.Clear;
end;

procedure DebugWrite(s:string);
begin
  //MainForm.DebugMemo.Append(s);
end;

procedure DumpGLScene(scene:TGLScene);
  var
    indent:integer;
  procedure DumpNode(node:TGLBaseSceneObject);
  var
    i: Integer;
    p: TGLCoordinates;
    proxy: tglproxyobject;
  begin
    p:=node.Position;
    DebugWrite(StringOfChar(' ',indent)+Format('%s:%s (%s) <%f,%f,%f> [',[hexStr(node),node.ClassName,node.Name,p.x,p.y,p.z]));
    Inc(indent,2);
    if node is tglproxyobject then begin
      proxy:=node as tglproxyobject;
      DebugWrite(StringOfChar(' ',indent)+'MasterObject: ');
      DumpNode(proxy.MasterObject);
    end;
    for i:=0 to node.Count-1 do begin
      DumpNode(node.Children[i]);
    end;
    Dec(indent,2);
    DebugWrite(StringOfChar(' ',indent)+']');
  end;
var
  root: TGLSceneRootObject;
begin
  root:=scene.Objects;
  indent:=0;
  DebugWrite('GLSCene.Objects dump');
  DumpNode(root);
end;

function Vector2Str(v:TVector):string;
begin
  Result:=Format('<%f, %f, %f; %f>',[v[0],v[1],v[2],v[3]]);
end;

{ TMainForm }

procedure TMainForm.PropHookModified(Sender: TObject);
begin
  //DebugLn('PropHookModified(Sender: 0x%X)',[PtrUInt(Sender)]);
end;

procedure TMainForm.PropHookRevert(Instance: TPersistent; PropInfo: PPropInfo);
begin
  //DebugLn('PropHookRevert(Instance: 0x%X, GetNamePath = %s; PropInfo: Name = %s)',[PtrUInt(Instance), Instance.GetNamePath, PropInfo^.Name]);
end;

procedure TMainForm.PropHookRefreshPropertyValues;
begin
  //DebugLn('PropHookRefreshPropertyValues()',[]);
end;

procedure TMainForm.PropHookPersistentAdded(APersistent: TPersistent; Select: boolean);
begin
  //DebugLn('PropHookPersistentAdded(APersistent: %s, Select: %s)',[APersistent.GetNamePath,BooleanIdents[Select]]);
  if APersistent is TVVariableFractal then RefreshVVFList;
end;

procedure TMainForm.PropHookPersistentDeleting(APersistent: TPersistent);
var
  Item: TListItem;
begin
  //DebugLn('PropHookPersistentDeleting(APersistent: %s)',[APersistent.GetNamePath]);
  if APersistent is TVVariableFractal then begin
    // delete from VVFListView
    Item:=VVFListView.Items.FindData(APersistent);
    if Assigned(Item) then begin
      VVFListView.Items.Delete(Item.Index);
    end;
  end;
end;

procedure TMainForm.PropHookComponentRenamed(AComponent: TComponent);
var
  Item: TListItem;
begin
  Item:=VVFListView.Items.FindData(AComponent);
  if Assigned(Item) then Item.Caption:=AComponent.Name;
end;

procedure TMainForm.SetLambda(const AValue: single);
var
  ClippedValue: Single;
begin
  ClippedValue:=Min(Max(AValue,0),1);
  if FLambda=ClippedValue then exit;
  // apply lambda
  FLambda:=ClippedValue;
  DebugLn('FLambda = %f; FUpdatingVVF = %s; FFVariableFractal.Initialized = %s',[FLambda,BooleanIdents[FUpdatingVVF],BooleanIdents[FVVariableFractal.Initialized]]);
  if not FUpdatingVVF and Assigned(FVVariableFractal) and FVVariableFractal.Initialized then begin
    ApplyLambda;
    UpdateVVFView(false);
  end;
  // update trackbar
  LambdaTrackBar.Link.LoadFromProperty;
end;

function TMainForm.GetLambdaInt: TLambdaInt;
begin
  Result:=Round(FLambda*High(TLambdaInt));
end;

procedure TMainForm.SetBufferDistance(const AValue: single);
begin
  if FBufferDistance=AValue then exit;
  FBufferDistance:=AValue;
  UpdateVVFView(false);
end;

procedure TMainForm.SetLambdaInt(const AValue: TLambdaInt);
begin
  Lambda:=AValue/High(TLambdaInt);
end;

procedure TMainForm.SetOpenFileName(const AValue: string);
begin
  // TODO
end;

procedure TMainForm.SetStickyDistance(const AValue: single);
begin
  if FStickyDistance=AValue then exit;
  FStickyDistance:=AValue;
  UpdateVVFView(false);
end;

procedure TMainForm.SetStickyMode(const AValue: TStickyMode);
begin
  if FStickyMode=AValue then Exit;
  FStickyMode:=AValue;
  StickyModeTIComboBox.Link.LoadFromProperty;
  UpdateVVFView(false);
end;

procedure TMainForm.ApplyLambda;
var
  vvf: TVVariableFractal;
  i: Integer;
begin
  vvf:=FVVariableFractal;
  if not Assigned(vvf) or not vvf.Initialized then Exit;
  case AnimationMode of
    amLast:begin
      vvf.Render(Lambda);
    end;
    amFirst:begin
      vvf.Render(0,Lambda);
      for i:=1 to vvf.Iteration-1 do
        vvf.Render(i,1);
    end;
    amAll:begin
      for i:=0 to vvf.Iteration-1 do
        vvf.Render(i,Lambda);
    end;
  end;
end;

procedure TMainForm.VVFListViewColumnClick(Sender: TObject; Column: TListColumn);
var
  ListView: TListView;
begin
  ListView:=Sender as TListView;
  if Assigned(Column) then begin
    ListView.BeginUpdate;
    ListView.SortColumn:=Column.Index;
    ListView.EndUpdate;
  end;
end;

procedure TMainForm.VVFListViewDblClick(Sender: TObject);
begin
  EditAction.Execute;
end;

procedure TMainForm.VVFListViewSelectItem(Sender: TObject; Item: TListItem; Selected: Boolean);
var
  VVF: TVVariableFractal;
begin
  VVF:=GetSelectedVVF;
  if Assigned(VVF) then begin
    FVVariableFractal:=VVF;
    UpdateVVFView(false);
  end;
end;

function TMainForm.GetSelectedVVF: TVVariableFractal;
var
  Item: TListItem;
begin
  Item:=VVFListView.Selected;
  if not Assigned(Item) then Exit(nil);
  Result:=TObject(Item.Data) as TVVariableFractal;
end;


// NB BackupFileName(FileName,0) = FileName
function BackupFileName(FileName:string; index:integer):string;
begin
  if Index=0 then Exit(FileName);
  Result:=Format('%s.bak.%d',[FileName, index]);
end;

// returns: BackupFileName(FileName,index)
function MoveBackupFileAway(FileName:string; index:integer):string;
const
  MaxIndex = 10;
begin
  Result:=BackupFileName(FileName,index);
  if FileExists(Result) then begin
    if index>=MaxIndex then begin
      DeleteFile(Result);
    end else begin
      RenameFile(Result,MoveBackupFileAway(FileName,index+1));
    end;
  end;
end;

procedure TMainForm.SaveDataModule(const FileName:string; DataModule:TDataModule);
var
  TempName: string;
  stream: TFileStream;
begin
  // write to temporary file
  TempName:=GetTempFileName('.', 'vvf');
  stream:=TFileStream.Create(TempName, classes.fmCreate);
  try
    WriteComponentAsTextToStream(stream,DataModule);
    //stream.WriteComponent(DataModule);
    // truncate FileMenuItem
    //stream.Size:=stream.Position;
  finally
    stream.Free;
  end;

  // writing succeeded; now start renaming chain
  MoveBackupFileAway(FileName, 0);
  RenameFile(TempName, FileName);
end;

function TMainForm.LoadDataModule(const FileName:string): TDataModule;
var
  stream: TFileStream;
begin
  Result:=nil;

  stream:=TFileStream.Create(FileName,classes.fmOpenRead);
  try
    ReadComponentFromTextStream(stream,Result,@FindComponentClass,nil,nil);
    //Result:=stream.ReadComponent(nil) as TDataModule;
  finally
    stream.Free;
  end;
end;

const
  VVFFile = 'vvariablefractal.vvf';

procedure TMainForm.FormCreate(Sender: TObject);
var
  root: TDataModule;
  tetraTop, tetraBack, tetraLeft, tetraRight: TVector;
  ContractTopHalf,  ContractBackHalf,  ContractLeftHalf,  ContractRightHalf:  TPointContraction;
  ContractTopThird, ContractBackThird, ContractLeftThird, ContractRightThird: TPointContraction;
  IFSHalf, IFSThird: TIFS;
  SceneBufFactory: T3DGLSceneBufferFactory;
  UniformIID: TIIDUniformSelector;
  stream:TStream;
  i: Integer;
  VVF: TVVariableFractal;
begin
  // register property editors
  //   use TPersistentPropertyEditor instead of TComponentPropertyEditor because we have no Designer and hence TComponentPropertyEditor.AllEqual returns false
  RegisterPropertyEditor(ClassTypeInfo(TComponent),nil,'',TPersistentPropertyEditor);
  //   register our TComponentReferencePropertyEditor, which restricts the Component references to ComponentClass
  RegisterPropertyEditor(ClassTypeInfo(TComponent),TComponentReferenceItem,'Component',TComponentReferencePropertyEditor);
  //   register TGLCoordinates property editor etc.
  GLLazarusRegister.Register;
  //   hide Tag editor
  RegisterPropertyEditor(TypeInfo(LongInt),TComponent,'Tag',nil);

  // register classes for streaming
  RegisterClasses([
    TDataModule,
    TVVariableFractal,
    TIFS,
    TTransformation,TCompoundTransformation,TPointContraction,TTranslation,TRotation,TScaling,
    TSelector,TIIDUniformSelector,TIIDWeightedSelector,
    TBufferFactory,T3DGLSceneBufferFactory]);

  root:=nil;
  if FileExists(VVFFile) then begin
    try
      // try to load FDataModule from VVFFile
      root:=LoadDataModule(VVFFile);
      FOpenFileName:=VVFFile;
    except
      on E:Exception do begin
        ShowMessageFmt('Error loading file %s: %s',[VVFFile,E.Message]);
      end;
    end;
  end;
  
  if not Assigned(root) then begin
    // create FDataModule & basic VVariableFractal
    root:=TDataModule.Create(nil);;

    // create a basic FVVariableFractal
    tetraTop  :=VectorMake( 0,   sqrt(6)/4,  0);
    tetraBack :=VectorMake( 0,  -sqrt(6)/12,-1/sqrt(3));
    tetraLeft :=VectorMake(-1/2,-sqrt(6)/12,1/sqrt(12));
    tetraRight:=VectorMake( 1/2,-sqrt(6)/12,1/sqrt(12));

    ContractTopHalf  :=TPointContraction.Create(root,tetraTop  , 1/2); ContractTopHalf.  Name:='ContractTopToHalf';
    ContractBackHalf :=TPointContraction.Create(root,tetraBack , 1/2); ContractBackHalf. Name:='ContractBackToHalf';
    ContractLeftHalf :=TPointContraction.Create(root,tetraLeft , 1/2); ContractLeftHalf. Name:='ContractLeftToHalf';
    ContractRightHalf:=TPointContraction.Create(root,tetraRight, 1/2); ContractRightHalf.Name:='ContractRightToHalf';

    ContractTopThird  :=TPointContraction.Create(root,tetraTop  , 1/3); ContractTopThird.  Name:='ContractTopToThird';
    ContractBackThird :=TPointContraction.Create(root,tetraBack , 1/3); ContractBackThird. Name:='ContractBackToThird';
    ContractLeftThird :=TPointContraction.Create(root,tetraLeft , 1/3); ContractLeftThird. Name:='ContractLeftToThird';
    ContractRightThird:=TPointContraction.Create(root,tetraRight, 1/3); ContractRightThird.Name:='ContractRightToThird';

    IFSHalf :=TIFS.Create(root,[ContractTopHalf,  ContractBackHalf,  ContractLeftHalf,  ContractRightHalf]);
    IFSHalf.Name :='SierpinskiContractToHalf';

    IFSThird:=TIFS.Create(root,[ContractTopThird, ContractBackThird, ContractLeftThird, ContractRightThird]);
    IFSThird.Name:='SierpinskiContractToThird';

    SceneBufFactory:=T3DGLSceneBufferFactory.Create(root);
    SceneBufFactory.Name:='Default3DGLSceneBufFactory';

    UniformIID:=TIIDUniformSelector.Create(root);
    UniformIID.Name:='UniformIID';

    VVF:=TVVariableFractal.Create(
      root, // component owner
      5, // number of buffers
      SceneBufFactory, // buffer factory
      [IFSHalf, IFSThird], // IFSs
      UniformIID, // buffer selector
      UniformIID  // IFS selector
    );
    VVF.Name:='MixedSierpinski';

    VVF:=TVVariableFractal.Create(
      root, // component owner
      1, // number of buffers
      SceneBufFactory, // buffer factory
      [IFSHalf], // IFSs
      UniformIID, // buffer selector
      UniformIID  // IFS selector
    );
    VVF.Name:='Sierpinski';
  end;
  
  // store laoded DataModule in FDataModule
  FDataModule:=root;

  // select first TVVariableFractal available
  for i:=0 to root.ComponentCount-1 do
    if root.Components[i] is TVVariableFractal then begin
      FVVariableFractal:=root.Components[i] as TVVariableFractal;
      Break;
    end;

  // assure availability of PropertyHook to property editors which use PropEdits.GlobalPropertyHook
  FPropertyEditorHook:=TPropertyEditorHook.Create;
  FPropertyEditorHook.LookupRoot:=root;
  FPropertyEditorHook.AddHandlerModified(@PropHookModified);
  FPropertyEditorHook.AddHandlerRevert(@PropHookRevert);
  FPropertyEditorHook.AddHandlerRefreshPropertyValues(@PropHookRefreshPropertyValues);
  FPropertyEditorHook.AddHandlerPersistentAdded(@PropHookPersistentAdded);
  FPropertyEditorHook.AddHandlerPersistentDeleting(@PropHookPersistentDeleting);
  FPropertyEditorHook.AddHandlerComponentRenamed(@PropHookComponentRenamed);
  FPropertyEditorHook.SelectOnlyThis(FVVariableFractal);

  GlobalDesignHook:=FPropertyEditorHook;

  // assign it to the property grid
  //TIPropertyGrid.PropertyEditorHook:=FPropertyEditorHook;

  // Object inspector
  (*OI:=TObjectInspector.Create(nil);
  OI.PropertyEditorHook:=GlobalDesignHook;
  OI.EnsureVisible(true);
  OI.Height:=800;
  OI.OnCloseQuery:=@OICloseQuery;*)

  // Object Editor
  FObjectEditor:=TObjectEditorForm.Create(nil);
  FObjectEditor.EnsureVisible(true);
  FObjectEditor.AddClasses([
    TVVariableFractal,
    TIFS,
    TTransformation,TCompoundTransformation,TPointContraction,TTranslation,TRotation,TScaling,
    TSelector,TIIDUniformSelector,TIIDWeightedSelector,
    TBufferFactory,T3DGLSceneBufferFactory]);
  FObjectEditor.PropertyEditorHook:=FPropertyEditorHook;
  
  // About Form
  FAboutForm:=TAboutForm.Create(nil);
  
  // Fill list of VVFs
  RefreshVVFList;
  
  // set properties
  FLambda:=1;
  FAnimate:=true;
  FBufferDistance:=1;
  FStickyDistance:=1;
  FStickyMode:=smNone;
  
  // Manual changes to propertylinks
  LambdaTrackBar.           Link.SetObjectAndProperty(Self,'LambdaInt');
  AnimateCheckBox.          Link.SetObjectAndProperty(Self,'Animate');
  BufDistTIFloatSpinEdit.   Link.SetObjectAndProperty(Self,'BufferDistance');
  StickyDistTIFloatSpinEdit.Link.SetObjectAndProperty(Self,'StickyDistance');
  StickyModeTIComboBox.     Link.SetObjectAndProperty(Self,'StickyMode');
  AniModeTIComboBox.        Link.SetObjectAndProperty(Self,'AnimationMode');
end;

procedure TMainForm.AboutMenuItemClick(Sender: TObject);
begin
  FAboutForm.ShowModal;
end;

procedure TMainForm.FindComponentClass(Reader: TReader; const AClassName: string; var ComponentClass: TComponentClass);
begin
  // use component registration system from Classes unit
  ComponentClass:=TComponentClass(GetClass(AClassName));
end;

procedure TMainForm.EditActionExecute(Sender: TObject);
begin
  // select FVVariableFractal in Object Editor (if any selected)
  if Assigned(FVVariableFractal) then begin
    FObjectEditor.PropertyEditorHook.SelectOnlyThis(FVVariableFractal);
  end;
  // make Object Editor visible (anyway)
  FObjectEditor.EnsureVisible(true);
end;

(*procedure TMainForm.TestButtonClick(Sender: TObject);
var
  s: TStringStream;
  NewRoot: TComponent;
begin
  s:=TStringStream.Create('');
  try
//    WriteComponentAsTextToStream(s,FRootComponent);
    s.WriteComponent(FDataModule);
    ShowMessage('Ready to detach RootComponent?');//s.DataString);
    GlobalDesignHook.SetSelection(TPersistentSelectionList.Create());
    FreeAndNil(FDataModule);
    ShowMessage('Ready to read RootComponent?');//s.DataString);
    s.Position:=0;
//    ReadComponentFromTextStream(s,nil,@FindComponentClass);
    FDataModule:=s.ReadComponent(nil) as TDataModule;
    GlobalDesignHook.SelectOnlyThis(FDataModule);
  finally
    s.Free;
  end;
//  ReadComponentFromTextStream();
end;*)

procedure TMainForm.FormDestroy(Sender: TObject);
var
  index: Integer;
  SaveFileName: String;
begin
  // make sure we first free the ObjectEditor
  // otherwise an ValueComboBox with Focus will trigger OnExit, which does a SetRowValue, but the data has been deleted!
  FreeAndNil(FObjectEditor);
  FreeAndNil(FAboutForm);
  
  GlobalDesignHook.RemoveAllHandlersForObject(self);

  if OpenFileName='' then
    SaveFileName:=VVFFile
  else
    SaveFileName:=OpenFileName;
  SaveDataModule(SaveFileName,FDataModule);
  
  FreeAndNil(FDataModule);
end;

procedure TMainForm.NavigationCustomAction(Sender: TGLSimpleNavigationKeyCombination; Shift: TShiftState; X, Y: Integer);
var
  DeltaX: Integer;
  DeltaY: Integer;
begin
  if ([ssLeft,ssRight]*Shift)=[ssLeft] then begin
    DeltaX:=X-FOldNavX;
    DeltaY:=Y-FOldNavY;

    GLScene.Objects.Translate(0.002*DeltaX,-0.002*DeltaY,0);
  end;
  
  FOldNavX:=X;
  FOldNavY:=Y;
end;

procedure TMainForm.GLCadencerProgress(Sender: TObject; const deltaTime, newTime: Double);
begin
  if Animate then
    Lambda:=Lambda+deltaTime;
end;

procedure TMainForm.StepActionExecute(Sender: TObject);
begin
  if Assigned(FVVariableFractal) then begin
    if not FVVariableFractal.Initialized then begin
      InitAction.Execute();
    end else begin
      FUpdatingVVF:=true;
      if (AnimationMode=amLast) and (Lambda<1) then
        FVVariableFractal.Render();
      FVVariableFractal.Iterate(false);
      if Animate then
        Lambda:=0
      else
        Lambda:=1;
      ApplyLambda;
      UpdateVVFView(false);
      FUpdatingVVF:=false;
    end;
  end;
end;

procedure TMainForm.SaveActionExecute(Sender: TObject);
begin
  SaveDataModule(OpenFileName,FDataModule);
end;

procedure TMainForm.ExitActionExecute(Sender: TObject);
begin
  if MessageDlg('Confirmation','Quit application?',mtConfirmation,mbYesNo,0) = mrYes then begin
    Application.Terminate;
  end;
end;

procedure TMainForm.InitActionExecute(Sender: TObject);
begin
  if Assigned(FVVariableFractal) then begin
    FUpdatingVVF:=true;
    FVVariableFractal.Init();
    FUpdatingVVF:=false;
    UpdateVVFView(true);
  end;
end;

procedure TMainForm.FormCloseQuery(Sender: TObject; var CanClose: boolean);
begin
  ExitAction.Execute;
  CanClose:=false;
end;

procedure TMainForm.UpdateVVFView(resetView:boolean);
var
  root: TGLSceneRootObject;
  proxy: TGLProxyObject;
  light: TGLBaseSceneObject;
  vvf: TVVariableFractal;
  bufCount: Integer;
  buf: TBuffer;
  bufGL: T3DGLSceneBuffer;
  i: Integer;
  target: TGLDummyCube;
  j: Integer;
  Grove: PGrove;
  itCount: LongInt;
begin
  vvf:=FVVariableFractal;
  if not Assigned(vvf) or not vvf.Initialized then Exit;
  GLScene.BeginUpdate;
  try
    // get scene root element
    root:=GLScene.Objects;
    root.DeleteChildren;
    // add a light
    light:=root.AddNewChild(TGLLightSource);
    light.Position.SetPoint(2,5,1);
    // add buffers of v-variable fractal
    bufCount:=vvf.BufferCount;
    itCount:=vvf.Iteration;
    for i:=0 to bufCount-1 do begin
      buf:=vvf.Buffer[i];
      if (buf is T3DGLSceneBuffer) then begin
        bufGL:=buf as T3DGLSceneBuffer;
        proxy:=bufGL.GetProxy();
        proxy.Translate(BufferDistance*i,0,0);
        root.AddChild(proxy);
      end;
      // add sticky buffers
      if StickyMode<>smNone then begin
        for j:=0 to itCount-1 do begin
          Grove:=vvf.Grove[j];
          buf:=Grove^.InputBuffers[i];
          if (buf is T3DGLSceneBuffer) then begin
            bufGL:=buf as T3DGLSceneBuffer;
            proxy:=bufGL.GetProxy();
            case StickyMode of
              smIdentity: proxy.Translate(BufferDistance*i,0,0);
              smMove:     proxy.Translate(BufferDistance*i,0,StickyDistance*(itCount-j));
            end;
            root.AddChild(proxy);
          end;
        end;
      end;
    end;
    // add a dummy target to look at
    target:=TGLDummyCube.Create(nil);
    target.Position.SetPoint((bufCount-1)/2,0,0);
    root.AddChild(target);
    // target the camera
    GLCamera.TargetObject:=target;
    if resetView then begin;
      GLCamera.Position.SetPoint((bufCount-1)/2,6,8);
      GLCamera.SceneScale:=5;
    end;
  finally
    GLScene.EndUpdate;
  end;
end;

procedure TMainForm.RefreshVVFList;
var
  i: Integer;
  Child: TComponent;
  VVF: TVVariableFractal;
  NewItem: TListItem;
begin
  DebugLn('TMainForm.RefreshVVFList');
  VVFListView.BeginUpdate;
  VVFListView.Clear;
  try
    for i:=0 to FDataModule.ComponentCount-1 do begin
      Child:=FDataModule.Components[i];
      if Child is TVVariableFractal then begin
        VVF:=Child as TVVariableFractal;
        NewItem:=VVFListView.Items.Add;
        NewItem.Caption:=VVF.Name;
        NewItem.Data:=VVF;
        NewItem.SubItems.Add(IntToStr(VVF.IFSCount));
      end;
    end;
  finally
    VVFListView.EndUpdate;
  end;
end;

{ TTransformation }

function TTransformation.GetMatrix: TMatrix;
begin
  //DebugLn('%s: GetMatrix; FNeedsCalcMatrix = %s',[Name,BooleanIdents[FNeedsCalcMatrix]]);
  if FNeedsCalcMatrix then begin
    // we set the flag to false to avoid infinite recursion
    FNeedsCalcMatrix:=false;
    CalcMatrix(1,FMatrix);
  end;
  Result:=FMatrix;
end;

procedure TTransformation.TransformationChanged;
begin
  if FNeedsCalcMatrix then Exit;
  //DebugLn('%s: TransformationChanged',[Name]);
  FNeedsCalcMatrix:=true;
  if Assigned(FChangeNotifyList) then
    FChangeNotifyList.CallNotifyEvents(self); // Note: TTransformationChangeEvent is compatible with TNotifyEvent
end;

destructor TTransformation.Destroy;
begin
  inherited Destroy;
  // Free this list only after FreeNotifications have been fired (by the inherited TComponent destructor)
  FChangeNotifyList.Free;
end;

procedure TTransformation.AddChangeNotification(AEvent:TTransformationChangeEvent);
begin
  if not Assigned(FChangeNotifyList) then
    FChangeNotifyList:=TMethodList.Create;
  FChangeNotifyList.Add(TMethod(AEvent));
  //DebugLn('%s: AddChangeNotification for %s',[Name,TComponent(TMethod(AEvent).Data).Name]);
end;

procedure TTransformation.RemoveChangeNotification(AEvent:TTransformationChangeEvent);
begin
  if not Assigned(FChangeNotifyList) then Exit;
  FChangeNotifyList.Remove(TMethod(AEvent));
  //DebugLn('%s: RemoveChangeNotification for %s',[Name,TComponent(TMethod(AEvent).Data).Name]);
end;

function TTransformation.MatrixInterp(lambda: single): TMatrix;
begin
  CalcMatrix(lambda, Result);
end;

{ TIFS }

function TIFS.GetTransformation(i: integer): TTransformation;
begin
  Result := FTransforms[i].Component as TTransformation;
end;

function TIFS.GetTransformationCount: integer;
begin
  Result := FTransforms.Count;
end;

procedure TIFS.SetTransformation(i: integer; const AValue: TTransformation);
begin
  FTransforms[i].Component:=AValue;
end;

procedure TIFS.SetTransforms(const AValue: TComponentReferences);
begin
  if FTransforms=AValue then Exit;
  FTransforms.Assign(AValue);
end;

constructor TIFS.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  FTransforms:=TComponentReferences.Create(self,TTransformation);
end;

constructor TIFS.Create(AOwner:TComponent; const transforms: array of TTransformation);
var
  i: Integer;
begin
  Create(AOwner);
  for i:=0 to High(transforms) do
    FTransforms.Add.Component:=transforms[i];
end;

destructor TIFS.Destroy;
begin
  FTransforms.Free;
  inherited;
end;

{ TVVariableFractal }

function TVVariableFractal.GetBuffer(i: integer): TBuffer;
begin
  Result:=FBuffers[i];
end;

function TVVariableFractal.GetBufferCount: integer;
begin
  Result:=Length(FBuffers);
end;

function TVVariableFractal.GetGrove(AIteration: integer): PGrove;
begin
  Result:=PGrove(FGroveList[AIteration]);
end;

function TVVariableFractal.GetIFS(i: integer): TIFS;
begin
  Result:=FIFSs[i].Component as TIFS;
end;

function TVVariableFractal.GetIFSCount: integer;
begin
  Result:=FIFSs.Count;
end;

function TVVariableFractal.GetInputBuffer(AIteration, i: integer): TBuffer;
begin
  Result:=Grove[AIteration]^.InputBuffers[i];
end;

function TVVariableFractal.GetInputBuffers(AIteration: integer): TBufferArray;
begin
  Result:=Grove[AIteration]^.InputBuffers;
end;

function TVVariableFractal.GetOutputBuffer(AIteration, i: integer): TBuffer;
begin
  Inc(AIteration);
  if AIteration<Iteration then
    Result:=Grove[AIteration]^.InputBuffers[i]
  else
    Result:=Buffer[i];
end;

function TVVariableFractal.GetOutputBuffers(AIteration: integer): TBufferArray;
begin
  Inc(AIteration);
  if AIteration<Iteration then
    Result:=Grove[AIteration]^.InputBuffers
  else
    Result:=FBuffers;
end;

procedure TVVariableFractal.SetBufferCount(const AValue: integer);
var
  i: Integer;
begin
  if Length(FBuffers)=AValue then Exit;
  // release current buffers
  FreeBuffers;
  // resize FBuffers
  SetLength(FBuffers, AValue);
  // need init; will re-create buffers
  FInitialized:=false;
end;

procedure TVVariableFractal.SetIFSs(const AValue: TComponentReferences);
begin
  FIFSs.Assign(AValue);
end;

procedure TVVariableFractal.FreeBuffers;
var
  i: Integer;
  TheGrove: PGrove;
begin
  for i:=0 to Iteration-1 do begin
    TheGrove:=Grove[i];
    FBufferFactory.ReleaseBuffers(TheGrove^.InputBuffers);
    TheGrove^.InputBuffers:=nil;
  end;
  FBufferFactory.ReleaseBuffers(FBuffers);
end;

function TVVariableFractal.AddGrove(): PGrove;
begin
  New(Result);
  FGroveList.Add(Result);
end;

function TVVariableFractal.LastGrove(): PGrove;
begin
  Result:=Grove[Iteration-1];
end;

procedure TVVariableFractal.FreeGroves;
var
  i: Integer;
  TheGrove: PGrove;
begin
  // free groves
  for i:=0 to FGroveList.Count-1 do begin
    TheGrove:=Grove[i];
    Dispose(TheGrove);
  end;
  // clear grove list
  FGroveList.Clear;
end;

constructor TVVariableFractal.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  FIFSs:=TComponentReferences.Create(self,TIFS);
  FGroveList:=TFPList.Create;
  FInitialShape:=shTetraeder;
end;

constructor TVVariableFractal.Create(AOwner:TComponent; bufCount: integer; const bufFactory: TBufferFactory; const ifsArray: array of TIFS; const bufSel, ifsSel: TSelector);
var
  i: Integer;
begin
  Create(AOwner);
  SetLength(FBuffers, bufCount);
  for i:=0 to High(ifsArray) do begin
    FIFSs.Add.Component:=ifsArray[i];
  end;
  FBufferFactory:=bufFactory;
  FBufferSelector:=bufSel;
  FIFSSelector   :=ifsSel;
end;

destructor TVVariableFractal.Destroy;
begin
  FreeBuffers;
  FreeGroves;
  FGroveList.Free;
  FIFSs.Free;
  inherited;
end;

procedure TVVariableFractal.Init();
var
  i: Integer;
begin
  DebugLn('%s: Init()',[Name]);
  // free current buffers & groves
  FreeBuffers;
  FreeGroves;
  // reset iteration count
  FIteration:=0;
  // set up initial buffers
  BufferFactory.GetBuffers(FBuffers);
  for i:=0 to High(FBuffers) do begin
    FBuffers[i].Initial(i,InitialShape,InitialTransform);
  end;
  FInitialized:=true;
end;

procedure TVVariableFractal.Iterate(doRender:boolean);
var
  NewGrove: PGrove;
  outBuffers:TBufferArray;
  IFSchoices, inputChoices: TIntegerArray;
  theIFS: TIFS;
  outBuf: TBuffer;
  i,j:integer;
begin
  DebugLn('%s: Iterate(doRender: %s)',[Name,BooleanIdents[doRender]]);
  // add grove
  NewGrove:=AddGrove();
  SetLength(NewGrove^.Trees,BufferCount);
  // store input buffers in grove
  NewGrove^.InputBuffers:=FBuffers;
  // create new output buffers
  SetLength(outBuffers,BufferCount);
  BufferFactory.GetBuffers(outBuffers);
  if IFSCount>0 then begin
    // choose IFSes for each output buffer
    IFSchoices:=IFSSelector.Sample(BufferCount,IFSCount);
    // store IFS choices in grove
    for i:=0 to BufferCount-1 do
      NewGrove^.Trees[i].IFSChoice:=IFSchoices[i];
    // fill buffer choices
    for i:=0 to High(outBuffers) do begin
      outBuf:=outBuffers[i];
      theIFS:=IFS[IFSchoices[i]];
      if not Assigned(theIFS) then Continue;
      if theIFS.TransformationCount<=0 then Continue;
      // choose input buffers for IFS components
      inputChoices:=BufferSelector.Sample(theIFS.TransformationCount,BufferCount);
      // store input choices in grove
      NewGrove^.Trees[i].BufferChoices:=inputChoices; // copy by reference is safe, as we don't write to inputBuffers anymore
    end;
  end;
  // replace buffers by output buffers
  FBuffers:=outBuffers;
  // increase iteration counter
  Inc(FIteration);
  // render iff requested
  if doRender then
    Render();
end;

procedure TVVariableFractal.Render(AIteration: integer; lambda: single);
var
  i, j: Integer;
  TheGrove: PGrove;
  TheTree: PFunctionTree;
  theIFS: TIFS;
  transform: TTransformation;
  inBufs, outBufs: TBufferArray;
begin
  DebugLn('%s: Render(AIteration = %d, lambda: %f)',[Name,AIteration,lambda]);
  if Iteration<=0 then Exit;
  TheGrove:=Grove[AIteration];
  inBufs:=InputBuffers[AIteration];
  outBufs:=OutputBuffers[AIteration];
  for i:=0 to Math.Min(High(TheGrove^.Trees), BufferCount-1) do begin
    // clear output buffer
    outBufs[i].Clear();
    // get tree & IFS
    TheTree:=@(TheGrove^.Trees[i]);
    if TheTree^.IFSChoice>=IFSCount then Continue;
    theIFS:=IFS[TheTree^.IFSChoice];
    if not Assigned(theIFS) then Continue;
    // for each buffer choice
    for j:=0 to Math.Min(High(TheTree^.BufferChoices), theIFS.TransformationCount-1) do begin
      transform:=theIFS.Transformation[j];
      if not Assigned(transform) then Continue;
      outBufs[i].AddTransformedBuf(transform, lambda, inBufs[TheTree^.BufferChoices[j]]);
    end;
  end;
  DebugLn('%s: Render: done',[Name]);
end;

procedure TVVariableFractal.Render(lambda: single);
begin
  Render(Iteration-1, lambda);
end;

procedure TVVariableFractal.Render();
begin
  Render(1);
end;

{ TIIDUniformSelector }

function TIIDUniformSelector.Sample(count, maxVal: integer): TIntegerArray;
var
  i: Integer;
begin
  SetLength(Result,count);
  for i:=0 to count-1 do begin
    Result[i]:=Random(maxVal);
  end;
end;

{ T3DGLSceneBuffer }

function T3DGLSceneBuffer.GetProxy(): TGLLimitProxyObject;
begin
  Result:=TGLLimitProxyObject.Create(nil);
  Result.MasterObject:=FObject;
//  Result.MinimalRenderSize:=0.01;
  Inc(FProxyCount);
end;

constructor T3DGLSceneBuffer.Create(ABufferFactory:T3DGLSceneBufferFactory);
begin
  FBufferFactory:=ABufferFactory;
  FObject:=TGLDummyCube.Create(nil);
  FProxyCount:=0;
end;

destructor T3DGLSceneBuffer.Destroy;
begin
  FObject.Free;
  inherited Destroy;
end;

procedure T3DGLSceneBuffer.Clear();
begin
  FObject.DeleteChildren;
end;

procedure T3DGLSceneBuffer.Initial(index:integer; shape:T3DShape; transform:TTransformation);
const
  Colors:array[0..5] of TColorVector = (
   (1.0,0.0,0.0,1.0),
   (0.0,1.0,0.0,1.0),
   (0.0,0.0,1.0,1.0),
   (0.0,1.0,1.0,1.0),
   (1.0,0.0,1.0,1.0),
   (1.0,1.0,0.0,1.0)
  );
var
  child: TGLCustomSceneObject;
  tetraeder: TGLCone;
  cube: TGLCube;
  line: TGLLines;
  color: TVector4f;
begin
  color:=Colors[index mod Length(Colors)];
  // create shape
  case shape of
    shTetraeder:begin
      tetraeder:=TGLCone.Create(nil);
      tetraeder.Height:=sqrt(6)/3;
      tetraeder.BottomRadius:=1/sqrt(3);
      tetraeder.Slices:=3;
      tetraeder.Stacks:=1;
      tetraeder.Translate(0,sqrt(6)/12,0); // make sure the (0,0,0) point is at the center of the tetraeder
      child:=tetraeder;
    end;
    shCube:begin
      cube:=TGLCube.Create(nil);
      child:=cube;
    end;
    shLine:begin
      line:=TGLLines.Create(nil);
      line.AddNode(0,-0.5,1);
      line.AddNode(0, 0.5,1);
      line.SplineMode:=lsmLines;
      line.NodesAspect:=lnaInvisible;
      line.LineColor.Color:=color;
      child:=line;
    end;
  end;
  // set color
  child.Material.FrontProperties.Diffuse.Color:=color;
  // set initial transformation
  if Assigned(transform) then
    child.Matrix:=transform.Matrix;
  // add to FObject
  FObject.AddChild(child);
end;

procedure T3DGLSceneBuffer.AddTransformedBuf(const transform: TTransformation; const lambda: single; const buffer: TBuffer);
var
  buf: T3DGLSceneBuffer;
  proxy: tglproxyobject;
begin
  buf:=buffer as T3DGLSceneBuffer;
  proxy:=buf.GetProxy();
  FObject.AddChild(proxy);
  proxy.Matrix:=transform.MatrixInterp(lambda);
end;

{ TBufferFactory }

procedure TBufferFactory.GetBuffers(var bufArray: TBufferArray);
var
  i: Integer;
begin
  for i:=0 to High(bufArray) do begin
    bufArray[i]:=GetBuffer();
  end;
end;

procedure TBufferFactory.ReleaseBuffers(bufs: TBufferArray);
var
  i: Integer;
begin
  for i:=0 to High(bufs) do begin
    ReleaseBuffer(bufs[i]);
  end;
end;

{ TCompoundTransformation }

procedure TCompoundTransformation.TransformationAdded(ACompRefs: TComponentReferences; AComponent:TComponent);
begin
  TransformationChanged;
  (AComponent as TTransformation).AddChangeNotification(@ATransformationChanged);
end;

procedure TCompoundTransformation.TransformationRemoving(ACompRefs: TComponentReferences; AComponent:TComponent);
begin
  (AComponent as TTransformation).RemoveChangeNotification(@ATransformationChanged);
  TransformationChanged;
end;

procedure TCompoundTransformation.SetTransformations(const AValue: TComponentReferences);
begin
  if FTransformations=AValue then exit;
  FTransformations.Assign(AValue);
end;

procedure TCompoundTransformation.CalcMatrix(lambda:single; out Mat:TMatrix);
var
  Transformation: TTransformation;
  i: Integer;
begin
  Mat:=IdentityHmgMatrix;
  for i:=0 to Transformations.Count-1 do begin
    Transformation:=Transformations[i].Component as TTransformation;
    if Assigned(Transformation) then
      Mat:=MatrixMultiply(Transformation.MatrixInterp(lambda), Mat);
  end;
end;

procedure TCompoundTransformation.ATransformationChanged(Sender: TObject);
begin
  //DebugLn('%s: ATransformationChanged from %s',[Name,(Sender as TComponent).Name]);
  TransformationChanged;
end;

constructor TCompoundTransformation.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FTransformations:=TComponentReferences.Create(Self,TTransformation);
  FTransformations.OnReferenceAdded   :=@TransformationAdded;
  FTransformations.OnReferenceRemoving:=@TransformationRemoving;
end;

destructor TCompoundTransformation.Destroy;
begin
  FTransformations.Free;
  inherited Destroy;
end;

{ TPointContraction }

procedure TPointContraction.SetFactor(const AValue: single);
begin
  //DebugLn('TPointContraction.SetFactor(%f)',[AValue]);
  if FFactor=AValue then Exit;
  FFactor:=AValue;
  TransformationChanged;
end;

procedure TPointContraction.SetPoint(const AValue: TGLCoordinates);
begin
  //DebugLn('TPointContraction.SetPoint(%f, %f, %f)',[AValue.X,AValue.Y,AValue.Z]);
  if FPoint.Equals(AValue.AsVector) then Exit;
  FPoint.AsVector:=AValue.AsVector;
end;

procedure TPointContraction.CalcMatrix(lambda:single; out Mat:TMatrix);
var
  AFactor: Extended;
begin
  AFactor:=lambda*Factor+(1-lambda)*1;
  Mat:=CreateScaleAndTranslationMatrix(VectorMake(AFactor,AFactor,AFactor),VectorScale(Point.AsVector,1-AFactor));
end;

procedure TPointContraction.CoordinateChanged(Sender: TGLCoordinates);
begin
  if Sender=FPoint then
    TransformationChanged;
end;

constructor TPointContraction.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  // init to identity
  FPoint:=TGLCoordinates.CreateInitialized(Self,NullHmgPoint,csPoint);
  FFactor:=1;
  TransformationChanged;
end;

constructor TPointContraction.Create(AOwner:TComponent; aPoint: TVector; aFactor: single);
begin
  Create(AOwner);
  // init to params
  FPoint.AsVector:=aPoint;
  FFactor:=aFactor;
  TransformationChanged;
end;

destructor TPointContraction.Destroy;
begin
  FPoint.Free;
  inherited Destroy;
end;

{ TTranslation }

procedure TTranslation.SetTranslation(const AValue: TGLCoordinates);
begin
  if FTranslation.Equals(AValue.AsVector) then exit;
  FTranslation.AsVector:=AValue.AsVector;
end;

procedure TTranslation.CalcMatrix(lambda:single; out Mat:TMatrix);
//var
//  i: Integer;
begin
  Mat:=CreateTranslationMatrix(VectorScale(Translation.AsVector,lambda));
//  for i:=0 to 3 do
//    DebugLn('Matrix[][%d]: %8f %8f %8f %8f',[i,FMatrix[0][i],FMatrix[1][i],FMatrix[2][i],FMatrix[3][i]]);
end;

procedure TTranslation.CoordinateChanged(Sender: TGLCoordinates);
begin
  if Sender=FTranslation then
    TransformationChanged;
end;

constructor TTranslation.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FTranslation:=TGLCoordinates.CreateInitialized(Self,NullHmgVector,csVector);
  TransformationChanged;
end;

destructor TTranslation.Destroy;
begin
  FTranslation.Free;
  inherited Destroy;
end;

{ TRotation }

procedure TRotation.SetAxis(const AValue: TGLCoordinates);
begin
  if FAxis.Equals(AValue.AsVector) then Exit;
  FAxis.AsVector:=AValue.AsVector;
end;

procedure TRotation.SetAngle(const AValue: single);
begin
  if FAngle=AValue then Exit;
  FAngle:=AValue;
  TransformationChanged;
end;

procedure TRotation.CalcMatrix(lambda:single; out Mat:TMatrix);
begin
  Mat:=CreateRotationMatrix(Axis.AsVector,DegToRad(lambda*Angle));
end;

procedure TRotation.CoordinateChanged(Sender: TGLCoordinates);
begin
  TransformationChanged;
end;

constructor TRotation.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  FAxis:=TGLCoordinates.CreateInitialized(self,ZHmgVector,csVector);
  TransformationChanged;
end;

destructor TRotation.Destroy;
begin
  FAxis.Free;
  inherited Destroy;
end;

{ TScaling }

procedure TScaling.SetScale(const AValue: TGLCoordinates);
begin
  if FScale.Equals(AValue.AsVector) then Exit;
  FScale.AsVector:=AValue.AsVector;
  TransformationChanged;
end;

procedure TScaling.CalcMatrix(lambda: single; out Mat: TMatrix);
begin
  Mat:=CreateScaleMatrix(VectorAdd(VectorScale(AffineVectorMake(1,1,1),1-lambda),
                                   VectorScale(FScale.AsAffineVector,lambda)));
end;

procedure TScaling.CoordinateChanged(Sender: TGLCoordinates);
begin
  TransformationChanged;
end;

constructor TScaling.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FScale:=TGLCoordinates.CreateInitialized(Self,VectorMake(1,1,1),csUnknown);
end;

destructor TScaling.Destroy;
begin
  FScale.Free;
  inherited Destroy;
end;

{ T3DGLSceneBufferFactory }

constructor T3DGLSceneBufferFactory.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
//  DebugLn('T3DGLSceneBufferFactory Created');
  FProxiedBufs:=TFPObjectList.Create(true);
end;

destructor T3DGLSceneBufferFactory.Destroy;
begin
  FProxiedBufs.Free; // also frees contained objects as we set FreeObjects := true
  inherited Destroy;
end;

function T3DGLSceneBufferFactory.GetBuffer(): TBuffer;
begin
  Result:=T3DGLSceneBuffer.Create(Self);
  Inc(FBufferCount);
//  DebugLn('T3DGLSceneBuffer Requested nr. %d',[FBufferCount]);
end;

procedure T3DGLSceneBufferFactory.ReleaseBuffer(buf: TBuffer);
var
  glBuf: T3DGLSceneBuffer;
begin
  if buf=nil then Exit;
  glBuf:=buf as T3DGLSceneBuffer;
  if glBuf.ProxyCount<=0 then begin
    glBuf.Free;
  end else begin
    FProxiedBufs.Add(glBuf);
  end;
  Dec(FBufferCount);
  if FBufferCount<=0 then begin
    FProxiedBufs.Clear; // also frees contained objects because we set FreeObjects = true
    FBufferCount:=0;
  end;
end;

{ TIIDWeightedSelector }

procedure TIIDWeightedSelector.SetWeights(w: TIntegerArray);
var
  i: Integer;
begin
  FWeights:=w;
  FTotalWeight:=0;
  for i:=0 to High(w) do begin
    Inc(FTotalWeight,w[i]);
  end;
end;

function TIIDWeightedSelector.Sample(count, maxVal: integer): TIntegerArray;
var
  RandomVals:TIntegerArray;
  i,j: Integer;
  sum: Integer;
begin
  SetLength(Result,count);
  SetLength(RandomVals,count);
  for i:=0 to count-1 do begin
    RandomVals[i]:=Random(FTotalWeight);
    Result[i]:=0;
  end;
  sum:=0;
  for j:=0 to Min(maxVal,Length(FWeights))-1 do begin
    Inc(sum,FWeights[j]);
    for i:=0 to count-1 do begin
      if sum>=RandomVals[i] then
        Result[i]:=j;
    end;
  end;
end;

{ TRootComponent }

procedure TRootComponent.SetRootInt(const AValue: integer);
begin
  if FRootInt=AValue then exit;
  FRootInt:=AValue;
end;

function ComponentName(C:TComponent):string; begin if C=nil then Result:='<nil>' else Result:=C.Name end;

procedure TRootComponent.SetChildOne(const AValue: TChildOne);
begin
//  DebugLn('TRootComponent.SetChildOne sets component %s',[ComponentName(AValue)]);
//  DebugLn(GetStackTrace(false));
  if FChildOne=AValue then exit;
  FChildOne:=AValue;
end;

procedure TRootComponent.SetChilds(const AValue: TComponentList);
begin
  if FChilds=AValue then exit;
  FChilds:=AValue;
end;

function TRootComponent.GetChildOne: TChildOne;
begin
  Result:=FChildOne;
//  DebugLn('TRootComponent.GetChildOne returns component %s',[ComponentName(Result)]);
end;

function TRootComponent.GetChildTwo: TChildTwo;
begin
  Result:=FChildTwo;
//  DebugLn('TRootComponent.GetChildTwo returns component %s',[ComponentName(Result)]);
end;

procedure TRootComponent.SetChildCollection(const AValue: TComponentReferences);
begin
  if FChildCollection=AValue then exit;
  FChildCollection:=AValue;
end;

procedure TRootComponent.SetChildTwo(const AValue: TChildTwo);
begin
//  DebugLn('TRootComponent.SetChildTwo sets component %s',[ComponentName(AValue)]);
  if FChildTwo=AValue then exit;
  FChildTwo:=AValue;
end;

procedure TRootComponent.SetRootString(const AValue: string);
begin
  if FRootString=AValue then exit;
  FRootString:=AValue;
end;

constructor TRootComponent.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FChilds:=TComponentList.Create(false);
  FChildCollection:=TComponentReferences.Create(self,TChildOne);
end;

destructor TRootComponent.Destroy;
begin
  inherited Destroy;
end;

procedure TRootComponent.GetChildren(Proc: TGetChildProc; Root: TComponent);
var
  i: Integer;
begin
  //inherited GetChildren(Proc, Root);
  // list all owned components
  for i:=0 to ComponentCount-1 do begin
    Proc(Components[i]);
  end;
(*  for i:=0 to Childs.Count-1 do begin
    Proc(Childs[i].Component);
  end;*)
end;

{ TChildOne }

procedure TChildOne.SetChildOneInt(const AValue: integer);
begin
  if FChildOneInt=AValue then exit;
  FChildOneInt:=AValue;
end;

procedure TChildOne.SetChildOneString(const AValue: string);
begin
  if FChildOneString=AValue then exit;
  FChildOneString:=AValue;
end;

procedure TChildOne.SetChildTwo(const AValue: TChildTwo);
begin
  if FChildTwo=AValue then exit;
  FChildTwo:=AValue;
end;

procedure TChildOne.GetChildren(Proc: TGetChildProc; Root: TComponent);
var
  i: Integer;
begin
  //inherited GetChildren(Proc, Root);
  // list all owned components
  for i:=0 to ComponentCount-1 do begin
    Proc(Components[i]);
  end;
end;

{ TChildTwo }

procedure TChildTwo.SetChildTwoInt(const AValue: integer);
begin
  if FChildTwoInt=AValue then exit;
  FChildTwoInt:=AValue;
end;

procedure TChildTwo.SetChildTwoString(const AValue: string);
begin
  if FChildTwoString=AValue then exit;
  FChildTwoString:=AValue;
end;

{ TComponentReferences }

function TComponentReferences.GetItem(index: integer): TComponentReferenceItem;
begin
  Result:=inherited GetItem(index) as TComponentReferenceItem;
end;

procedure TComponentReferences.SetItem(index: integer; const AValue: TComponentReferenceItem);
begin
  inherited SetItem(index, AValue);
end;

procedure TComponentReferences.NotifyReferenceAdded(AComponent: TComponent);
begin
  if Assigned(OnReferenceAdded) then OnReferenceAdded(self,AComponent);
end;

procedure TComponentReferences.NotifyReferenceRemoving(AComponent: TComponent);
begin
  if Assigned(OnReferenceRemoving) then OnReferenceRemoving(self,AComponent);
end;

constructor TComponentReferences.Create(AOwner: TPersistent);
begin
  Create(AOwner,TComponent);
  FComponentClass:=TComponent;
  //DebugLn('ComponentClass = %s',[ComponentClass.ClassName]);
end;

constructor TComponentReferences.Create(AOwner: TPersistent; AComponentClass: TComponentClass);
begin
  inherited Create(Owner,TComponentReferenceItem);
  FFreeNotificationSink:=TFreeNotificationSink.Create(nil,Self);
  FComponentClass:=AComponentClass;
  //DebugLn('ComponentClass = %s',[ComponentClass.ClassName]);
end;

destructor TComponentReferences.Destroy;
begin
  inherited Destroy;
  FFreeNotificationSink.Free;
end;

function TComponentReferences.Add: TComponentReferenceItem;
begin
  Result:=TComponentReferenceItem.Create(self);
end;

{ TComponentReferenceItem }

function TComponentReferenceItem.GetDisplayName: string;
begin
  if Assigned(Component) then
    Result:=Component.Name
  else
    Result:=inherited GetDisplayName;
end;

procedure TComponentReferenceItem.AttachComponent;
begin
  if Assigned(FComponent) and Assigned(FComponent) then begin
    FComponent.FreeNotification(Collection.FreeNotificationSink);
    Collection.NotifyReferenceAdded(FComponent);
  end;
end;

procedure TComponentReferenceItem.DetachComponent;
begin
  if Assigned(FComponent) and Assigned(Collection) then begin
    Collection.NotifyReferenceRemoving(FComponent);
    FComponent.RemoveFreeNotification(Collection.FreeNotificationSink);
  end;
end;

procedure TComponentReferenceItem.Assign(source: TPersistent);
begin
  if Source is TComponentReferenceItem then begin
    Component:=(Source as TComponentReferenceItem).Component;
  end else
    inherited Assign(source);
end;

procedure TComponentReferenceItem.SetComponent(const AValue: TComponent);
begin
  if FComponent=AValue then exit;
  //if AValue<>nil then DebugLn('SetComponent(%s: %s); ComponentClass = %s',[AValue.Name,AValue.ClassName,ComponentClass.ClassName]);
  if Assigned(AValue) and not (AValue is ComponentClass) then
    raise Exception.CreateFmt('Cannot assign a %s because the Component reference is restricted to %s',[AValue.ClassName, ComponentClass.ClassName]);
  //DebugLn('No exception eoccured');
  DetachComponent;
  FComponent:=AValue;
  AttachComponent;
  Changed(false);
end;

function TComponentReferenceItem.GetComponentClass: TComponentClass;
begin
  if Assigned(Collection) then
    Result:=Collection.ComponentClass
  else
    Result:=TComponent;
end;

function TComponentReferenceItem.GetCollection: TComponentReferences;
begin
  Result:=inherited Collection as TComponentReferences;
end;

procedure TComponentReferenceItem.SetCollection(const AValue: TComponentReferences);
begin
  if AValue=Collection then Exit;
  DetachComponent;
  inherited Collection:=AValue;
  AttachComponent;
end;

constructor TComponentReferenceItem.Create(AOwner: TCollection);
begin
  inherited Create(AOwner);
end;

destructor TComponentReferenceItem.Destroy;
begin
  DetachComponent;
  inherited Destroy;
end;

{ TComponentReferencePropertyEditor }

procedure TComponentReferencePropertyEditor.GetValues(Proc: TGetStringProc);
var
  ComponentReferenceItem: TComponentReferenceItem;
  Instance: TPersistent;
begin
  Instance:=GetComponent(0); // get the ComponentReferenceItem instance
  if Instance is TComponentReferenceItem then begin
    ComponentReferenceItem:=Instance as TComponentReferenceItem;
    Proc(oisNone);
    if Assigned(PropertyHook) then
      PropertyHook.GetComponentNames(GetTypeData(ClassTypeInfo(ComponentReferenceItem.ComponentClass)), Proc);
  end else
    inherited GetValues(Proc);
end;

{ TFreeNotificationSink }

procedure TFreeNotificationSink.Notification(AComponent: TComponent; Operation: TOperation);
var
  CompRef: TComponentReferenceItem;
  i: Integer;
begin
  case Operation of
    opRemove:
      for i:=0 to Collection.Count-1 do begin
        CompRef:=Collection[i];
        if CompRef.Component=AComponent then
          CompRef.Component:=nil;
      end;
  end;
  inherited Notification(AComponent, Operation);
end;

constructor TFreeNotificationSink.Create(AOwner: TComponent; ACollection: TComponentReferences);
begin
  inherited Create(AOwner);
  FCollection:=ACollection;
end;

{ TComponentListPropertyEditor }

function TComponentListPropertyEditor.GetComponentList: TComponentList;
begin
  Result:=GetObjectValue as TComponentList;
end;

function TComponentListPropertyEditor.ReadElementCount: integer;
var
  ComponenList: TComponentList;
begin
  ComponenList:=GetComponentList;
  if Assigned(ComponenList) then
    Result:=ComponenList.Count
  else
    Result:=0;
end;

function TComponentListPropertyEditor.ReadElement(Index: integer): TPersistent;
var
  ComponenList: TComponentList;
begin
  ComponenList:=GetComponentList;
  if Assigned(ComponenList) then
    Result:=ComponenList[Index]
  else
    Result:=nil;
end;

function TComponentListPropertyEditor.GetElementAttributes(Element: TListElementPropertyEditor): TPropertyAttributes;
begin
  Result:=inherited GetElementAttributes(Element);
end;

function TComponentListPropertyEditor.GetElementName(Element: TListElementPropertyEditor): shortstring;
begin
  Result:=(GetElement(Element) as TComponent).Name;
end;

procedure TComponentListPropertyEditor.GetElementProperties(Element: TListElementPropertyEditor; Proc: TGetPropEditProc);
begin
  inherited GetElementProperties(Element, Proc);
end;

function TComponentListPropertyEditor.GetElementValue(Element: TListElementPropertyEditor): ansistring;
begin
  Result:=inherited GetElementValue(Element);
end;

procedure TComponentListPropertyEditor.GetElementValues(Element: TListElementPropertyEditor; Proc: TGetStringProc);
begin
  inherited GetElementValues(Element, Proc);
end;

procedure TComponentListPropertyEditor.SetElementValue(Element: TListElementPropertyEditor; NewValue: ansistring);
begin
  inherited SetElementValue(Element, NewValue);
end;

function TComponentListPropertyEditor.GetAttributes: TPropertyAttributes;
begin
  Result:=[paDialog, paReadOnly];
end;

procedure TComponentListPropertyEditor.Edit;
begin
  inherited Edit;
end;

{ TGLLimitProxyObject }

constructor TGLLimitProxyObject.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FMinimalRenderSize:=0; // by default we render everything (ok, unless it is really crushed)
end;

destructor TGLLimitProxyObject.Destroy;
begin
  inherited Destroy;
end;

procedure TGLLimitProxyObject.DoRender(var ARci: TRenderContextInfo; ARenderSelf, ARenderChildren: Boolean);
var
  ModelMat:TMatrix4f;
  colI: Integer;
  unitLength: Single;
begin
  glGetFloatv(GL_MODELVIEW_MATRIX,@ModelMat);
  for colI:=0 to 2 do begin
    // get length of colI'th unit vector in model space
    unitLength:=VectorLength(AffineVectorMake(ModelMat[colI]));
    // check length against MinimalRenderSize
    if unitLength>=MinimalRenderSize then begin
      inherited DoRender(ARci, ARenderSelf, ARenderChildren);
      Break; // do not render twice!
    end;
  end;
end;

initialization
  {$I vvariablefractalmain.lrs}

end.

