(*
 * V-variable Fractal Generator
 *
 * File:   objecteditor.pas
 * Author: Bram Kuijvenhoven (bram (at) kuijvenhoven (dot) net)
 * Descr.: Object editor tool window
 *
 *)

unit ObjectEditor;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, LResources, Forms, Controls, Graphics, Dialogs, ComCtrls,
  RTTIGrids, PropEdits, Menus, StdCtrls, ExtCtrls;

type

  { TObjectEditorForm }

  TObjectEditorForm = class(TForm)
    // controls
    ClassTreeView: TTreeView;
    InstanceListView: TListView;
    InstanceListViewPanel: TPanel;
    TIPropertyGrid: TTIPropertyGrid;
    // popup menus
    NodePopupMenu: TPopupMenu;
    AddMenuItem: TMenuItem;
    InstancePopupMenu: TPopupMenu;
    RemoveMenuItem: TMenuItem;
    // GUI event handlers
    procedure AddMenuItemClick(Sender: TObject);
    procedure ClassTreeViewContextPopup(Sender: TObject; MousePos: TPoint; var Handled: Boolean);
    procedure ClassTreeViewSelectionChanged(Sender: TObject);
    procedure FormClose(Sender: TObject; var CloseAction: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure InstanceListViewSelectItem(Sender: TObject; Item: TListItem; Selected: Boolean);
    procedure RemoveMenuItemClick(Sender: TObject);
  private
    FPropertyEditorHook: TPropertyEditorHook;
    FLookupRoot: TPersistent; // LookupRoot from FPropertyEditorHook
    FSelection: TPersistentSelectionList;
    FInstanceListClass:TClass;
    procedure SetLookupRoot(const AValue: TPersistent);
    procedure SetPropertyEditorHook(const AValue: TPropertyEditorHook);
    procedure SetSelection(const AValue: TPersistentSelectionList);
    { private declarations }
  protected
    function FindMostRecentAncestorNode(AClass:TClass):TTreeNode;
    function GetSelectedClass:TClass;         // get class selected in ClassTreeView
    function GetSelectedInstance:TPersistent; // get instance selected in InstanceListView
    function GetInstanceListClass:TClass;     // get class used in InstanceListView
    procedure SetSelectedClass(AClass:TClass; ChangeOnlyIfCurrentIsNotAClass:boolean);
    procedure SetSelectedInstance(AInstance:TPersistent);
    procedure RefreshInstanceList;
    procedure RefreshSelection;
    // property editor hooks
    procedure PropHookGetSelection(const ASelection: TPersistentSelectionList);
    procedure PropHookSetSelection(const ASelection: TPersistentSelectionList);
    procedure PropHookChangeLookupRoot;
    procedure PropHookRefreshPropertyValues;
    procedure PropHookPersistentAdded(APersistent: TPersistent; Select: boolean);
    procedure PropHookPersistentDeleting(APersistent: TPersistent);
    procedure PropHookComponentRenamed(AComponent: TComponent);
  public
    { public declarations }
    // constructor/destructor
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
    // methods
    procedure AddClass(AClass:TClass);
    procedure AddClasses(const AClass:array of TClass);
    // properties
    property LookupRoot:TPersistent read FLookupRoot write SetLookupRoot;
    property PropertyEditorHook:TPropertyEditorHook read FPropertyEditorHook write SetPropertyEditorHook;
    property Selection:TPersistentSelectionList read FSelection write SetSelection;
  end;

implementation

uses
  LCLProc;

procedure GiveName(Instance: TComponent);
var
  i: Integer;
  Prefix: String;
  Nr: Integer;
  ChildName: String;
  ChildNameSuffix: String;
  Owner: TComponent;
begin
  if not Assigned(Instance) or (Instance.Name<>'') then Exit;
  // determine Prefix & init Nr
  Prefix:=Copy(Instance.ClassName,2,MaxInt); // cut off the leading 'T'
  Nr:=1;
  Owner:=Instance.Owner;
  // try to find a unique name of the form PrefixNr
  if Assigned(Owner) then
    for i:=0 to Owner.ComponentCount-1 do begin
      ChildName:=Owner.Components[i].Name;
      if CompareText(Copy(ChildName,1,Length(Prefix)),Prefix)=0 then begin
        // prefix hit; update Nr if needed
        ChildNameSuffix:=Copy(ChildName,Length(Prefix)+1,MaxInt);
        Nr:=StrToIntDef(ChildNameSuffix,Nr-1)+1;
        //ebugLn('Found Child %s with suffix %s; bumped Nr to',[ChildName, ChildNameSuffix]);
      end
    end;
  // name it
  Instance.Name:=Prefix+IntToStr(Nr);
end;

function GetClassName(AClass:TClass):string;       begin if Assigned(AClass) then Result:=AClass.ClassName else Result:='<nil class>'; end;
function GetNodeClassName(ANode:TTreeNode):string; begin if Assigned(ANode)  then Result:=GetClassName(TClass(ANode.Data)) else Result:='<nil node>'; end;

{ TObjectEditorForm }

procedure TObjectEditorForm.ClassTreeViewContextPopup(Sender: TObject; MousePos: TPoint; var Handled: Boolean);
var
  TreeView: TTreeView;
  Node: TTreeNode;
begin
  TreeView:=Sender as TTreeView;
  Node:=TreeView.GetNodeAt(MousePos.x, MousePos.y);
  if Assigned(Node) then begin
    AddMenuItem.Enabled:=not Node.HasChildren;
    TreeView.PopupMenu:=NodePopupMenu;
  end else
    TreeView.PopupMenu:=nil;
end;

procedure TObjectEditorForm.ClassTreeViewSelectionChanged(Sender: TObject);
begin
  if GetSelectedClass<>GetInstanceListClass then
    RefreshInstanceList;
end;

procedure TObjectEditorForm.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
  CloseAction:=caHide;
end;

procedure TObjectEditorForm.FormCreate(Sender: TObject);
begin
end;

procedure TObjectEditorForm.FormDestroy(Sender: TObject);
begin
end;

procedure TObjectEditorForm.InstanceListViewSelectItem(Sender: TObject; Item: TListItem; Selected: Boolean);
var
  Instance: TPersistent;
begin
  if not Selected then Exit; // ignore selection changes from code
  // signal change to propertyhook
  Instance:=GetSelectedInstance;
  if Assigned(PropertyEditorHook) and Assigned(Instance) then
    PropertyEditorHook.SelectOnlyThis(Instance);
end;

procedure TObjectEditorForm.RemoveMenuItemClick(Sender: TObject);
var
  Instance: TObject;
  Persistent: TPersistent;
  Item: TListItem;
  NewSelection: TPersistentSelectionList;
  Index: LongInt;
begin
  // get selected item & instance
  Item:=InstanceListView.Selected;
  if not Assigned(Item) then Exit;
  Instance:=TObject(Item.Data);
  Index:=Item.Index;
  // delete instance
  if Assigned(Instance) and Assigned(PropertyEditorHook) and (Instance is TPersistent) then begin
    Persistent:=Instance as TPersistent;
    PropertyEditorHook.PersistentDeleting(Persistent);
    PropertyEditorHook.DeletePersistent(Persistent);
  end;
  // try to select other
  NewSelection:=TPersistentSelectionList.Create;
  try
    if Index>=InstanceListView.Items.Count then
      Index:=InstanceListView.Items.Count-1;
    if Index>=0 then
      NewSelection.Add(TPersistent(InstanceListView.Items[Index].Data));
    Selection:=NewSelection;
  finally
    NewSelection.Free;
  end;
end;

procedure TObjectEditorForm.SetPropertyEditorHook(const AValue: TPropertyEditorHook);
begin
  if FPropertyEditorHook=AValue then exit;
  // detach old hook
  if Assigned(FPropertyEditorHook) then begin
    // remove handlers from old property hook
    FPropertyEditorHook.RemoveAllHandlersForObject(Self);
  end;
  // set new property hook
  FPropertyEditorHook:=AValue;
  // attach new hook
  if Assigned(FPropertyEditorHook) then begin
    // add handlers
    FPropertyEditorHook.AddHandlerGetSelection(@PropHookGetSelection);
    FPropertyEditorHook.AddHandlerSetSelection(@PropHookSetSelection);
    FPropertyEditorHook.AddHandlerRefreshPropertyValues(@PropHookRefreshPropertyValues);
    FPRopertyEditorHook.AddHandlerChangeLookupRoot(@PropHookChangeLookupRoot);
    FPropertyEditorHook.AddHandlerPersistentAdded(@PropHookPersistentAdded);
    FPropertyEditorHook.AddHandlerPersistentDeleting(@PropHookPersistentDeleting);
    FPropertyEditorHook.AddHandlerComponentRenamed(@PropHookComponentRenamed);
    // set look up root
    LookupRoot:=FPropertyEditorHook.LookupRoot;
    // set selection
    FPropertyEditorHook.GetSelection(FSelection);
    RefreshSelection;
  end else begin
    LookupRoot:=nil;
  end;
  // apply to propertygrid
  TIPropertyGrid.PropertyEditorHook:=FPropertyEditorHook;
end;

procedure TObjectEditorForm.SetSelection(const AValue: TPersistentSelectionList);
begin
  if FSelection.IsEqual(AValue) then Exit;
  FSelection.Assign(AValue);
  RefreshSelection;
end;

function TObjectEditorForm.FindMostRecentAncestorNode(AClass: TClass): TTreeNode;
var
  Node, LastNode: TTreeNode;
begin
  // find most recent ancestor that is in the ClassTreeView
  Result:=nil;
  Node:=ClassTreeView.Items.GetFirstNode;
  while Assigned(Node) do begin
    if AClass.InheritsFrom(TClass(Node.Data)) then begin
      //DebugLn('%s inherits from %s',[GetClassName(AClass),GetNodeClassName(Node)]);
      Result:=Node;
      Node:=Node.GetFirstChild;
    end else begin
      Node:=Node.GetNextSibling;
    end;
  end;
end;

procedure TObjectEditorForm.SetLookupRoot(const AValue: TPersistent);
begin
  if FLookupRoot=AValue then Exit;
  FLookupRoot:=AValue;
  RefreshInstanceList;
end;

function TObjectEditorForm.GetSelectedClass: TClass;
var
  Node: TTreeNode;
begin
  Node:=ClassTreeView.Selected;
  if Assigned(Node) then
    Result:=TClass(Node.Data)
  else
    Result:=nil;
end;

function TObjectEditorForm.GetSelectedInstance: TPersistent;
begin
  Result:=TObject(InstanceListView.Selected.Data) as TPersistent;
end;

function TObjectEditorForm.GetInstanceListClass: TClass;
begin
  Result:=FInstanceListClass;
end;

procedure TObjectEditorForm.SetSelectedClass(AClass: TClass; ChangeOnlyIfCurrentIsNotAClass: boolean);
var
  CurrentClass: TCLass;
  Node: TTreeNode;
begin
  if not Assigned(AClass) then Exit;
  //DebugLn('TObjectEditorForm.SetSelectedClass(%s,%s)',[AClass.ClassName,BoolToStr(ChangeOnlyIfCurrentIsNotAClass,true)]);
  CurrentClass:=GetSelectedClass;
  //DebugLn('TObjectEditorForm.SetSelectedClass: CurrentClass = %s',[GetClassName(CurrentClass)]);
  // check if change is necessary
  if ChangeOnlyIfCurrentIsNotAClass and AClass.InheritsFrom(CurrentClass) then Exit;
  // find & select the right node
  Node:=FindMostRecentAncestorNode(AClass);
  //DebugLn('TObjectEditorForm.SetSelectedClass: Node.Data = %s',[GetNodeClassName(Node)]);
  if Assigned(Node) then Node.Selected:=true;
end;

procedure TObjectEditorForm.SetSelectedInstance(AInstance: TPersistent);
var
  Item: TListItem;
begin
  if AInstance=nil then Exit;
  Item:=InstanceListView.Items.FindData(AInstance);
  if Assigned(Item) then Item.Selected:=true;
end;

procedure TObjectEditorForm.RefreshInstanceList;
var
  TheClass: TClass;
  i: Integer;
  Component: TComponent;
  Child: TComponent;
  Item: TListItem;
begin
  InstanceListView.BeginUpdate;
  try
    // clear listbox
    InstanceListView.Items.Clear;
    // get selected class
    TheClass:=GetSelectedClass;
    FInstanceListClass:=TheClass;
    // fill listbox
    if Assigned(TheClass) and Assigned(PropertyEditorHook) and Assigned(LookupRoot) and (LookupRoot is TComponent) then begin
      Component:=LookupRoot as TComponent;
      for i:=0 to Component.ComponentCount-1 do begin
        Child:=Component.Components[i];
        if Child is TheClass then begin
          Item:=InstanceListView.Items.Add;
          Item.Data:=Child;
          Item.Caption:=Child.Name;
          Item.SubItems.Add(Child.ClassName);
        end;
      end;
    end;
    // refresh selection
    if Selection.Count>0 then
      SetSelectedInstance(Selection[0]);
  finally
    InstanceListView.EndUpdate;
  end;
end;

procedure TObjectEditorForm.RefreshSelection;
var
  Selected: TPersistent;
begin
  // get (first) selected item
  if FSelection.Count<=0 then
    Selected:=nil
  else
    Selected:=FSelection[0];
  // selected it
  if Assigned(Selected) then begin
    SetSelectedInstance(Selected);
    SetSelectedClass(Selected.ClassType,true);
    // apply to property grid
    DebugLn('TIPropertyGrid.Selection := %s',[Selected.GetNamePath]);
    TIPropertyGrid.Selection:=FSelection;
  end;
end;

procedure TObjectEditorForm.PropHookGetSelection(const ASelection: TPersistentSelectionList);
begin
  if not Assigned(ASelection) then Exit;
  ASelection.Assign(FSelection);
end;

procedure TObjectEditorForm.PropHookSetSelection(const ASelection: TPersistentSelectionList);
begin
  Selection:=ASelection;
end;

procedure TObjectEditorForm.PropHookChangeLookupRoot;
begin
  DebugLn('TIPropertyGrid.PropHookChangeLookupRoot',[]);
  TIPropertyGrid.PropEditLookupRootChange;
  if Assigned(FPropertyEditorHook) then
    LookupRoot:=FPropertyEditorHook.LookupRoot;
end;

procedure TObjectEditorForm.PropHookRefreshPropertyValues;
begin
  DebugLn('TIPropertyGrid.RefreshPropertyValues',[]);
  TIPropertyGrid.RefreshPropertyValues;
end;

procedure TObjectEditorForm.PropHookPersistentAdded(APersistent: TPersistent; Select: boolean);
var
  TheClass: TClass;
begin
  TheClass:=GetInstanceListClass;
  if Assigned(TheClass) and (APersistent is TheClass) then begin
    RefreshInstanceList;
  end;
end;

procedure TObjectEditorForm.PropHookPersistentDeleting(APersistent: TPersistent);
var
  Item: TListItem;
begin
  Item:=InstanceListView.Items.FindData(APersistent);
  if Assigned(Item) then
    Item.Delete;
end;

procedure TObjectEditorForm.PropHookComponentRenamed(AComponent: TComponent);
var
  Item: TListItem;
begin
  Item:=InstanceListView.Items.FindData(AComponent);
  if Assigned(Item) then Item.Caption:=AComponent.Name;
end;

constructor TObjectEditorForm.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FSelection:=TPersistentSelectionList.Create;
end;

destructor TObjectEditorForm.Destroy;
begin
  FreeAndNil(FSelection);
  inherited Destroy;
end;

procedure TObjectEditorForm.AddMenuItemClick(Sender: TObject);
var
  TheClass: TClass;
  Instance: TComponent;
  TheOwner: TComponent;
begin
  // determine which the Class of which an instance is to be created
  TheClass:=GetSelectedClass;
  // create instance (if possible)
  Instance:=nil;
  if Assigned(TheClass) and TheClass.InheritsFrom(TComponent) then begin
    // get PropertyEditorHook.LookupRoot as Owner for new TComponent instance
    if Assigned(PropertyEditorHook) then begin
      LookupRoot:=PropertyEditorHook.LookupRoot;
      if LookupRoot is TComponent then begin
        TheOwner:=LookupRoot as TComponent;
        Instance:=TComponentClass(TheClass).Create(TheOwner);
        // name it
        GiveName(Instance);
      end;
      if Assigned(Instance) then begin
        // add & select Instance (if created)
        PropertyEditorHook.PersistentAdded(Instance,false);
        PropertyEditorHook.SelectOnlyThis(Instance);
      end;
    end;
  end;
end;

procedure TObjectEditorForm.AddClass(AClass: TClass);
var
  Node: TTreeNode;
begin
  Node:=FindMostRecentAncestorNode(AClass);
  // check if already in TreeView
  if Assigned(Node) and (TClass(Node.Data)=AClass) then Exit;
  // add to LastNode
  ClassTreeView.BeginUpdate;
  ClassTreeView.Items.AddChildObject(Node,AClass.ClassName,AClass);
  ClassTreeView.EndUpdate;
end;

procedure TObjectEditorForm.AddClasses(const AClass: array of TClass);
var
  i: Integer;
begin
  ClassTreeView.BeginUpdate;
  for i:=Low(AClass) to High(AClass) do
    AddClass(AClass[i]);
  ClassTreeView.EndUpdate;
end;

initialization
  {$I objecteditor.lrs}

end.

