function obs = randobs(n, Tdist, Cdist);
% RANDOBS generates observation times for the CRIC problem
%   OBS = RANDOBS(N, T)
%     Repeat the observation T N times. N should be a scalar and T a vector.
%     The result obs will be a cell array with T repeated n times.
%
%   OBS = RANDOBS(N, TDIST, C)
%     Generate N sets of observations [T1 ... TC] with given C and given distribution of T.
%     N is a scalar; the number of observation sets to generate.
%     TDIST describes the distribution of each Ti, for i = 1:C, by P(Ti <= TDIST.x) = TDIST.F.
%     C should be a scalar: the number of observation times per observation.
%     OBS will be a cell array of vectors of length C, each being an observation [T1 ... TC].
%
%   OBS = RANDOBS(N, TDIST, CDIST)
%     Same as before, but CDIST descibes the distribution of C by P(C <= CDIST.x) = CDIST.F.

% input checks
if ~(nargin == 2 | nargin == 3); error('wrong number of parameters'); end
if ~isscalar(n); error('parameter N should be a scalar'); end
if nargin == 2;
    if ~isvector(Tdist); error('parameter T should be a vector'); end
else % nargin == 3
    [d, msg] = isdist(Tdist); if ~d; error(['parameter TDIST is invalid: ' msg]); end
    if isa(Cdist,'numeric')
        if ~isscalar(Cdist); error('parameter C is not a scalar'); end
    else
        [d, msg] = isdist(Cdist); if ~d; error(['parameter CDIST is invalid: ' msg]); end
    end
end
    
% handle OBS = RANDOBS(N, T)
if nargin == 2 
    obs = repmat({Tdist},1,n); % NOTE: the formal name of the second parameter is Tdist
    break
end

% generate the C's
if isa(Cdist,'numeric') % OBS = RANDOBS(N, TDIST, C)
    C = Cdist*ones(1, n); % NOTE: Cdist is the formal name for the third parameter
else % OBS = RANDOBS(N, TDIST, CDIST)
    C = round(draw(Cdist, n));
end

% generate the observation times
obs = cell(1,n);
for j = 1:n
    obs{j} = sort(draw(Tdist, C(j)));
end
