function sample = randcricsample(dists, weights, obs)
% RANDCRICSAMPLE generates random CRIC samples (Competing Risks Interval Censoring)
%   SAMPLE = RANDCRICSAMPLE(DISTS, WIEGHTS, OBS)
%     DISTS is an vector of structs that descibe the conditional subdistributions by
%       P(X <= DISTS(k).x | Y = k) = DISTS(k).F
%     where X is the failure time and Y the failure cause.
%     WEIGHTS is a vector of same length as DISTS, giving the probabilities for Y by
%       P(Y = k) = WEIGHTS(k)/sum(WEIGHTS(k))
%     OBS is a cell vector holding the observation times for each sample point
%     SAMPLE will be a struct array of the same size as OBS, with fields t1, t2, k1, k2 and K:
%       Let x be the generated failure time and y the generated failure cause, then
%         t1 and t2 are the measure times that lie around x, i.e. t1 < x <=t2, or undefined otherwise
%         k1 is 0 if there was a measure time t before x, i.e. t < x, or -1 otherwise
%         k2 is y if there was a measure time t after x, i.e. x >= t, or 0 otherwise
%       SAMPLE.K will hold the number of possible failure causes (which is the length of DISTS and WEIGHTS)

% input checks
if ~isvector(dists); error('parameter DISTS is not a vector (of distributions)'); end
for i = 1:length(dists);
    [d, msg] = isdist(dists(i));
    if ~d; error(sprintf('DISTS(%d) is not valid: %s',i,msg)); end
end
if ~isvector(weights); error('parameter WEIGHTS is not a vector'); end
if ~isnumeric(weights); error('parameter WEIGHTS is not numeric'); end
if length(weights) ~= length(dists); error('parameters DISTS and WEIGHTS are not of equal lengt.='); end
if ~isvector(obs); error('parameter OBS is not a (cell) vector'); end
if ~iscell(obs); error('parameter OBS is not a cell vector'); end

n = length(obs); % number of samples
K = length(dists); % number of possible failure causes

w = cumsum(colvec(weights));
w = w./w(K);

% decompose r into x and y (failure times and causes respectively)
% very smart interp1 usage for y :-)
y = ceil(interp1([0; w],0:K,rand(n,1),'linear','extrap'));
% now draw x
x = zeros(n,1);
for i = 1:n
    x(i) = draw(dists(y(i)), 1);
end

% construct t1, t2, k1, k2
t1 = zeros(n,1);
t2 = zeros(n,1);
k1 = zeros(n,1);
k2 = zeros(n,1);
for i = 1:n
    o = obs{i};
    failuretime = x(i);
    k = y(i);
    u = max(find(o < failuretime));
    v = min(find(failuretime <= o));
    if u ~= []
        t1(i) = o(u);
        k1(i) = 0;
    else
        t1(i) = -1;
        k1(i) = -1;
    end
    if v ~= []
        t2(i) = o(v);
        k2(i) = k;
    else
        t2(i) = -1;
        k2(i) = -1;
    end
end

sample = struct('t1',t1,'t2',t2,'k1',k1,'k2',k2,'K',K);

% statistics output

% plot(sort(x),(1:n)./n);
% axis([min(x) max(x) 0 1]);
 
for i = 1:K; freq(i) = length(find(y == i)); end
fprintf('failure cause statistics:\n');
fprintf('  k  freq\n');
fprintf('%3d %5d\n',[1:K; freq]);

fprintf('observation statistics:\n');
fprintf('                       total; k =');
for k = 1:K; fprintf(' %4d', k); end
fprintf('\n');

fprintf('  type ''X <= T1'':       %4d     ', length(find(k1 == -1)));
for k = 1:K; fprintf(' %4d', length(find(k1 == -1 & y == k))); end
fprintf('\n');

fprintf('  type ''T1 < X <= TC'':  %4d     ', length(find(k1 == 0 & k2 > 0)));
for k = 1:K; fprintf(' %4d', length(find(k1 == 0 & k2 > 0 & y == k))); end
fprintf('\n');

fprintf('  type ''TC < X'':        %4d     ', length(find(k2 == -1)));
for k = 1:K; fprintf(' %4d', length(find(k2 == -1 & y == k))); end
fprintf('\n');