function [d, msg] = isdist(dist)
% ISDIST checks whether a given variable describes a distribution
%   D = ISDIST(DIST) returns true if and only if DIST is a valid distribution, i.e.
%     - DIST is a struct array with fields x anf F
%     - both DIST.x and DIST.F are monotonically non-decreasing
%   [D, MSG] = ISDIST(DIST) also returns an error message in the MSG parameter if DIST is not a valid distribution

% assume it's not a valid distribution
d = 0;
% run checks
if ~isstruct(dist); msg = 'the parameter is not a valid distribution because it is not a struct'; break; end
if ~isfield(dist,'x'); msg = 'the parameter is not a valid distribution because it has no x field'; break; end
if ~isfield(dist,'F'); msg = 'the parameter is not a valid distribution because it has no F field'; break; end
if ~isnumeric(dist.x); msg = 'the parameter is not a valid distribution because the field x is not numeric'; break; end
if ~isnumeric(dist.F); msg = 'the parameter is not a valid distribution because the field F is not numeric'; break; end
if ~isvector(dist.x); msg = 'the parameter is not a valid distribution because the field x is not a vector'; break; end
if ~isvector(dist.F); msg = 'the parameter is not a valid distribution because the field F is not a vector'; break; end
if length(dist.x) ~= length(dist.F); msg = 'the parameter is not a valid distribution because the field x and the field F are not of equal length'; break; end
x = colvec(dist.x);
if any([-Inf; x] > [x; Inf]); msg = 'the parameter is not a valid distribution because the field x is not monotonically non-decreasing'; break; end
F = colvec(dist.F);
if any([-Inf; F] > [F; Inf]); msg = 'the parameter is not a valid distribution because the field F is not monotonically non-decreasing'; break; end
% dist is a valid distribution
msg = '';
d = 1;