function sample = draw(dist, n)
% DRAW draws a i.i.d. sample from a given distribution
%   SAMPLE = DRAW(DIST, N)
%     Draws a sample of length N from the distribution described by DIST.
%     DIST should be a structure with fields x and F.
%       x and F should be monotonically non-decreasing vectors of equal length.
%       They describe the distribution of the stochast X by: P(X <= x(i)) = F(i).
%     N should be a scalar.
%     SAMPLE will be a vector of length N: the sample drawn.

% input checks
[d, msg] = isdist(dist); if ~d; error(['parameter DIST is invalid: ' msg]); end
if ~isscalar(n); error('n is not a scalar'); end

% extract x and F
x = colvec(dist.x);
F = colvec(dist.F);

% random number generation
r = rand(n,1); % generate n numbers ~ U(0,1)

% determine sets of unique values in F
[uniqueF, I, J] = unique(F);
uniqueN = length(uniqueF);

if uniqueN == length(F)
    % the values in F are unique, so we can straight use interp1
    sample = interp1(F, x, r, 'linear', 'extrap');
else
    % the values in F are not unqie, so we can't directly use interp1
    
    % calculate the uniqueness sets of F
    % i.e. F(uniqueMin(i):uniqueMax(i)) are all the same unique value in F, for i = 1:uniqueN
	[dummy, uniqueMin, dummy] = unique(flipud(J));
	uniqueMin = length(J)+1 - uniqueMin;
	[dummy, uniqueMax, dummy] = unique(J);
	
	% determine for each random number the uniqueness sets of F they are between
	uniqueIndices = interp1(uniqueF, 1:uniqueN, r, 'linear', 'extrap'); % use interp1 for speed
	lower = floor(uniqueIndices);
	upper = ceil(uniqueIndices);
	% resolve cases where 0<min(F) or max(F)<1
    underflow = find(lower < 1);
	overflow = find(upper > uniqueN);
	lower(underflow) = 1;
	upper(underflow) = 2;
	lower(overflow) = uniqueN-1;
	upper(overflow) = uniqueN;
	
	% linear interpolation between the closest points in the uniqueness sets
	lowerx = x(uniqueMax(lower));
	upperx = x(uniqueMin(upper));
	sample = lowerx + ((upperx - lowerx)./(uniqueIndices - lower));
end