/*
 * File:        icm.h 
 * Author:      Bram Kuijvenhoven (bkuijvenhoven@student.tudelft.nl)
 * Date:        2005/04/09 [yyyy/mm/dd]
 * Description: Header file for icm.cpp
 */

#ifndef ICM_H
#define ICM_H

#include <stdlib.h>
#include <math.h>
#include <iostream>
#include <iomanip>

using namespace std;

class CICMSolver {

public:

	// parameters for algorithm
	double fAccuracy;
	int    fMaxIterations;

	// output control parameters
	bool     fFullOutput; // flag indicating whether to output only tables relevant for the solution, or also process info
	ostream *fOutput;     // ostream to write output to
	
	CICMSolver();  // constructor
	virtual ~CICMSolver(); // destructor

	void Solve(double **dist); // minimizes phi over the space of distributions, storing the result in dist
	
	double **AllocDist(); // allocates a distribution using fNumDists, fDistSize
	void     FreeDist(double **dist);  // deallocates a distribution using fNumDists

protected:

	bool fNeedLagrangian; // if true, a Lagrangian term is required in order to make sure F_{+,p} <= 1
	double fLambda; // lamda to use for Lagrangian term

	double fEpsilon; // used in Fenchel optimality check; derive it from fAccucary

	// parameters describing the dimensions of the distributions that are to be estimated
	int    fNumDists;  // the number of distributions to be estimated simultaneously
	int   *fDistSize;  // the number of points estimated per distribution

	// pure virtual abstract methods that should calculate phi and its derivatives
	virtual double Phi(double **dist) = 0;            
	virtual void GradPhi(double **dist, double **grad) = 0;        
	virtual void HessianDiagPhi(double **dist, double **hdiag) = 0;

	// pure virtual abstract method that should calculate an initial estimate
	virtual void InitialEstimate(double **dist) = 0;

	// methods called before/after Solve(); can be used to do some processing on the input/ouput of Solve()
	virtual void BeforeSolve();
	virtual void AfterSolve();

	// utility methods
	void WriteDist(ostream &os, double **dist);

private:

	bool FenchelOptimality(double **dist, double **grad);
	void GreatestConvexMinorant(double *G, double *V, int distSize, int *hull, double *dist);
	double LineSearch(double **dist, double **newDist, double **diffDist, double **tempGrad, double **tempDist);

	// util function used by linesearch
	void LinearCombination(double **dist, double **diffDist, double alpha, double **tempDist);

};

#endif
