/*
 * File:        cric.h 
 * Author:      Bram Kuijvenhoven (bkuijvenhoven@student.tudelft.nl)
 * Date:        2005/04/11 [yyyy/mm/dd]
 * Description: Header file for cric.cpp (Competing Risks Interval Censoring)
 */

#ifndef CRIC_H
#define CRIC_H

#include <iostream>
#include "icm.h"

/* TCRICSample: one sample point of a Competing Risks Interval Censoring sample
 *
 * Define X the failure time, Y the failure cause (taking values in the range 1...K)
 * Observation times are T_1 < ... < T_C
 * Define T_0 = -Infinity, T_{C+1} = Infinity
 * Define U, V and I by  T_{I-1} = U < X <= V = T_I
 * 
 * The following table describes the representation of this in TCRICSample 
 * 
 *    I  |  t1  |  t2  |  k1  |  k2  |
 * ------+------+------+------+------+
 *     1 |   *  |   V  |  -1  |   Y  |
 *  2..C |   U  |   V  |   0  |   Y  |
 *   C+1 |   U  |   *  |   0  |  -1  |
 * ------+------+------+------+------+
 * 
 * Here, * denotes any value
 * Actually k = 0 denotes 'no failure yet' and k = -1 'observation does not exist'
 */

typedef struct {
	double t1,t2;
	int    k1,k2;
} TCRICSample;

/* reading/writing TCRICSamples
 */
TCRICSample *ReadCRICSamples(istream &is, int &sampleSize, int &numCauses);
void WriteCRICSamples(ostream &os, TCRICSample *samples, int sampleSize, int numCauses);

/* types used by CCRICSolver
 */
typedef struct {
	int i; // time index
	int n; // number of times observed
} TNi;

typedef struct {
	int i; // time index 1
	int j; // time index 2
	int n; // number of times observed
} TNij;

class CCRICSolver: public CICMSolver {

public:

	void SetSample(TCRICSample *sample, int sampleSize, int numCauses); // call before Solve()

protected:

	// the unique observation times
	int     fTimeSize; // number of unique observation times
	double *fTime;     // unique observation times

	// lists of observations per kind (Ni, Nki, Nkij) for fast evaluation of phi and its derivatives
	//   time indices i and j are indices to the fTime array
	//   n indicates the number of sample points with the indicated observation
	//   note: the indices k are zero based, in contrast with TCRICSample.k1/k2 and range from 0 to fNumDists-1
	int     fNiSize;   // number of elements in fNi array
	TNi    *fNi;       // list of sample points with no failure observed at any observation
	int    *fNkiSize;  // fNkiSize[k]: number of elements in fNki[k] array
	TNi   **fNki;      // fNki[k]: list of sample points with failure k observed right at first observation
	int    *fNkijSize; // fNkijSize[k]: number of elements in fNkij[k] array
	TNij  **fNkij;     // fNkij[k]: list of sample points with failure k observed to be in between two observations

	// mapping from fTime indices to Dist indices
	int   **fDistIndex;     // fDistIndex[k][i]: index in Dists arrays corresponding with cause k, unique time i;
	                        // -1 if not appearing in Dists arrays for this k;
	                        // 0 <= k < fNumDists, 0 <= i < fTimeSize

	// methods that calculate phi and its derivatives
	double Phi(double **dist);            
	void GradPhi(double **dist, double **grad);        
	void HessianDiagPhi(double **dist, double **hdiag);

	// method that calculates an initial estimate
	void InitialEstimate(double **dist);

	// utility methods
	void WriteTables(ostream &os);

};

#endif
