/*
 * File:        comprisk.cpp 
 * Author:      Bram Kuijvenhoven (bkuijvenhoven@student.tudelft.nl)
 * Date:        2005/03/08 [yyyy/mm/dd]
 * Description: Main routine that combines it all
 */

#include <fstream>
#include <iostream>

#include "cric.h"
#include "stdlib.h"
#include "getopt.h"

int main(int argc, char *argv[]) {

	const char *getHelpMsg = "Type comprisk -h to get help on usage";

	// options
	char *inputFileName = NULL;
	char *outputFileName = NULL;
	bool help = false;
	double accuracy = 1e-3;
	bool fullOutput = false;
	int maxIters = 100;

	// parse commandline options, using the getopt lib
	int c;

	while ((c = getopt(argc, argv, "a:fi:n:o:h")) != -1) {
		switch (c) {
		case 'a':
			sscanf(optarg, "%lf", &accuracy);
			break;
		case 'f':
			fullOutput = true;
			break;
		case 'h':
			// help
			help = true;
			break;
		case 'i':
			// input file name
			inputFileName = optarg;
			break;
		case 'n':
			sscanf(optarg, "%d", &maxIters);
			break;
		case 'o':
			// output file name
			outputFileName = optarg;
			break;
		case '?':
			// error
			cerr << getHelpMsg << endl;
			return 1;
		}
	}

	if (optind < argc) {
		// additional options encountered; this is an error
		cerr << "The option " << argv[optind] << " is not a valid option" << endl << getHelpMsg << endl;
		return 1;
	}

	if (help) {
		cerr 
			<< "COMPRISK (C) 2005 Bram Kuijvenhoven (bkuijvenhoven@student.tudelft.nl)" << endl
			<< "Competing Risks Interval Censoring Solver" << endl
			<< endl
			<< "Usage: comprisk [OPTIONS]" << endl
			<< "Example: comprisk -i sample.txt -o solution.txt" << endl
			<< "" << endl
			<< "Options:" << endl
			<< "  -a ACCURACY Set accuracy" << endl
			<< "  -f          Enable full output (detailed process info)" << endl
			<< "  -h          Show this help text" << endl
			<< "  -i FILE     Read sample input from FILE" << endl
			<< "  -n ITERS    Set maximum number of iterations to ITERS" << endl
			<< "  -o FILE     Write solution output to FILE" << endl;
		return 0;
	}

	// input file name


	// input file stream object
	istream *inFile;
	if (inputFileName) {
		cerr << "Constructing inFile(" << inputFileName << ")" << endl;
		inFile = new ifstream(inputFileName, ios_base::in);
	} else {
		cerr << "Reading input from stdin" << endl;
		inFile = &cin;
	}
	
	cerr << "Reading sample data from inFile" << endl;

	// read input sample data
	int sampleSize;
	int numCauses;
	TCRICSample *samples = ReadCRICSamples(*inFile, sampleSize, numCauses);

	if (inputFileName) {
		cerr << "Destructing inFile" << endl;
		delete inFile;
	}

	// output file stream object
	ostream *outFile;
	if (outputFileName) {
		cerr << "Constructing outFile(" << outputFileName << ")" << endl;
		outFile = new ofstream(outputFileName, ios_base::out);
	} else {
		cerr << "Writing output to stdout" << endl;
		outFile = &cout;
	}

	cerr << "Constructing CRICSolver object" << endl;

	// solver object
	CCRICSolver CRICSolver;

	CRICSolver.fOutput = outFile;
	CRICSolver.fAccuracy = accuracy;
	CRICSolver.fFullOutput = fullOutput;
	CRICSolver.fMaxIterations = maxIters;
	
	cerr << "Sending sample data to the CRICSolver object" << endl;
	
	// feed our sample data to the solver
	CRICSolver.SetSample(samples, sampleSize, numCauses);

	cerr << "Allocating dist" << endl;

		// allocate space for the solution distribution
	double **dist = CRICSolver.AllocDist();

	cerr << "Solving" << endl;
	
	// solve it
	CRICSolver.Solve(dist);

	cerr << "Cleanup" << endl;

	// destruct the outFile iff allocated by myself
	if (outputFileName) { delete outFile; }

	// free the solution
	CRICSolver.FreeDist(dist);

	// free the sample data
	delete samples;

	return 0;
}
